require 'controls'
require 'vector'

class Player
  
  MAX_DIST = 0x7fffffff
  CENTER_WINDOW = 300*300
  TRACK_CURRENT_TARGET_DIST = 250*250
  TRACK_ANY_TARGET_DIST = 450*450
  MAX_TURN_ANGLE_DIFFERENCE = 200
  MAX_SHIP_SPEED = 4
  CRUISE_SHIP_SPEED = 2
  HYPERSPACE_SAFETY_TIME = 7
  HYPERSPACE_SAFETY_DIST = 32**2    # ship size ~25x25
  COLLISION_SAFETY_TIME = 75
  COLLATERAL_SHOT_LIMIT = 80 # use no more than n frames for collateral shots
  MAX_OBJECTS = 26 - 3
  TURN_LOOKAHEAD = 40
  READY_TO_FIRE_CORRECTION = 0 # how many frames before the calculated shot lifetime ends can we fire again 
  
  SHOTS_TILL_ANNIHILATION = {
    :small_asteroid  => 1,
    :medium_asteroid => 2,
    :large_asteroid  => 3,
    :small_saucer    => 1,
    :large_saucer    => 1
  }

  def hyperspace_necessary?(game)
    objects_to_consider = []
    objects_to_consider << game[:asteroids]
    objects_to_consider << game[:saucer]
    objects_to_consider << game[:shots].select{|s| s[:tag].to_s =~ /^u/}
    
    vx_ship, vy_ship = Vector.motion_prediction(game[:ship],2+game[:latency])
    distances_in_the_future = objects_to_consider.flatten.collect do |o|
      next if o.nil? || o.empty?

      vx, vy = Vector.motion_prediction(o,2+game[:latency])
      
      dist, dx, dy = Vector.distance(vx_ship, vy_ship, vx, vy, o[:r_coll])
      dist
    end

    (distances_in_the_future.compact.min || MAX_DIST) < HYPERSPACE_SAFETY_DIST # hyperspace collision predicted
  end
  
  def give_command
    game = @ringbuffer.get_last

    @controls.reset

    min_dist = MAX_DIST

    unless game[:enabled]
      @controls.start if $game_ready
      return
    end

    return unless game[:ship_present] # not much we can do without a ship

    @state[:strategy] = if $timelimit && game[:duration].to_i > $timelimit
      :suicide
    else
      :target
    end

    verify_blacklist!(game)
    target_object = nil

    if @state[:strategy] == :target
      locked_on_targets = game[:locked_on_targets]

      if !locked_on_targets.empty?
        vt = valid_targets(locked_on_targets)
        @state[:valid_targets] = vt
        unless vt.empty?
          # keep enough shots for the actual target!
          # spare_shots = target_object ? (!vt.include?(@chosen_target) ? (SHOTS_TILL_ANNIHILATION[target_object[:class]]||1) : 0) : 0
          # TODO: find out what is better /\  \/
          # keep one shot for the actual target!
          spare_shots = !vt.include?(@chosen_target) ? 1 : 0

          if ready_to_fire?(game,spare_shots) == 0
            # "the_target" is the closest target that has not been shot at (blacklisted)
            the_target = locked_on_targets.reject{|t| @target_blacklist.include?(t[:tag]) && @shots_at_target[t[:tag]] >= (SHOTS_TILL_ANNIHILATION[t[:class]]||1)}.min{|t1, t2| game[:objects_by_tag][t1[:tag]][:lock_on] <=> game[:objects_by_tag][t2[:tag]][:lock_on]}
            if the_target && vt.include?(the_target[:tag])
              fire_at(game,the_target[:tag])
              puts "[#{game[:sequence]}] fired at #{the_target[:tag] == @chosen_target ? 'CHOSEN' : 'en-passant'} target #{the_target[:tag]}" if $debug
            end
          end
        end
      end

      target_object = select_target_and_hyperspace_in_emergency!(game) # also adds angle_difference tags to asteroids!
      @chosen_target = target_object ? target_object[:tag] : nil

      # always fly at cruise speed to make shots faster and close in on targets
      # @controls.thrust unless (game[:ship][:mx].abs+game[:ship][:my].abs > CRUISE_SHIP_SPEED rescue true)

    elsif @state[:strategy] == :center
      angle_difference = game[:ship][:dx] * dy - game[:ship][:dy] * dx
      if angle_difference > 0
        @controls.left
      else
        @controls.right
      end
      @controls.thrust unless (game[:ship][:mx].abs+game[:ship][:my].abs > MAX_SHIP_SPEED rescue false)

    elsif @state[:strategy] == :stop
      angle_difference = (game[:ship][:dx] * -game[:ship][:my] - game[:ship][:dy] * -game[:ship][:mx] rescue 0)
      if angle_difference > 0
        @controls.left
      elsif angle_difference < 0
        @controls.right
      end
      @controls.thrust unless angle_difference.abs > MAX_TURN_ANGLE_DIFFERENCE || (game[:ship][:mx].abs+game[:ship][:my].abs < 1 rescue false) # stop ship

    elsif @state[:strategy] == :suicide
      # run into next best target to end the game as soon as possible
      unless @chosen_target
        target_object = game[:asteroids].first
        @chosen_target = target_object ? target_object[:tag] : nil
      else
        target_object = game[:objects_by_tag][@chosen_target]
        @chosen_target = target_object ? target_object[:tag] : nil
      end
      @controls.thrust
    end

    @state[:current_target] = @chosen_target

    if target_object
      angle_difference = target_object[:angle_difference] || target_angle(game, target_object)
    
      if angle_difference > 0
        @controls.left
      elsif angle_difference < 0
        @controls.right
      end

      # beschleunigen, wenn Ziel weit entfernt
      # if (@chosen_target == :ufo && target_object[:dist] > TRACK_CURRENT_TARGET_DIST) || target_object[:dist] > TRACK_ANY_TARGET_DIST 
      # if target_object[:dist] > TRACK_ANY_TARGET_DIST 
      #   @controls.thrust unless (game[:ship][:mx].abs+game[:ship][:my].abs > MAX_SHIP_SPEED rescue true)
      # end

    # elsif @state[:strategy] == :target || @state[:strategy] == :lurk # turn left if no target available
    #   # ... only if other targets exist
    #   @controls.left unless @target_blacklist.size >= (game[:asteroids].size + (game[:saucer_present] ? 1 : 0))
    end

  rescue # show exceptions without terminating player altogether
    puts $!.message if $debug
    puts $!.backtrace.join("\n") if $debug
  ensure
    controls_packet = game ? @controls.prepare_command(game[:packet_count]) : nil
    @state[:blacklist] = @target_blacklist
    @ringbuffer.attach_controls(@controls.status, controls_packet)
  end

  def initialize(ringbuffer, state, socket = nil)
    @ringbuffer = ringbuffer
    @ringbuffer.register_observer(self, :give_command)
    @controls = Controls.new(socket)
    @state = state
    @chosen_target = nil
    @target_blacklist = []
    @shots_at_target = {}
    @blacklist_time = {}
    @shots = []
  end

protected

  def verify_blacklist!(game)
    @shots_at_target.delete(@target_blacklist.shift) if @target_blacklist.size > 27 # max 26 asteroids + 1 saucer on screen

    # verify shots
    shots_to_be_deleted = []
    @shots.each do |s|
      if s[:processed] && game[:sequence]-s[:processed] >= s[:lifetime]
        shots_to_be_deleted << s
      elsif !s[:processed] && (game[:ping] > s[:ping] || (game[:ping] < s[:ping] && s[:ping] > 200 && game[:ping] < 55))
        shots_to_be_deleted << s
      end
      s[:processed] = game[:sequence] if game[:ping] == s[:ping]
    end
    shots_to_be_deleted.each do |s|
      # FIXME: targets that need multiple shots should not be removed completely, but have their
      #        shot counter decremented!?
      ex_shot = @shots.delete(s)
      remove_from_blacklist!(ex_shot[:target]) if ex_shot[:target]
    end
    
    # if only one target exists clear blacklist altogether
    if (game[:saucer_present] && game[:asteroids].size == 0) ||
       (!game[:saucer_present] && game[:asteroids].size == 1 && (SHOTS_TILL_ANNIHILATION[game[:asteroids].first[:class]]||1) == 1)
       @target_blacklist = []
       @shots_at_target  = {}
       @blacklist_time   = {}
       return
    end

    # reset blacklisting for ufo if none present
    remove_from_blacklist!(:ufo) unless game[:saucer_present] || @state[:strategy] != :target
  end
  
  def remove_from_blacklist!(tag)
    @target_blacklist.delete(tag)
    @shots_at_target.delete(tag)
    @blacklist_time.delete(tag)
  end
  
  def valid_targets(locked_on_targets)
    if @state[:strategy] == :lurk
      return (locked_on_targets.any?{|o| o[:tag] == :ufo} ? [:ufo] : [])
    end
    vt = []
    locked_on_targets.each do |t|
      tag = t[:tag]
      if @target_blacklist.include?(tag)
        if @shots_at_target[tag] < (SHOTS_TILL_ANNIHILATION[t[:class]]||1)
          vt << tag if t[:lock_on] < COLLATERAL_SHOT_LIMIT || tag == @chosen_target
        end
      else
        vt << tag if t[:lock_on] < COLLATERAL_SHOT_LIMIT || tag == @chosen_target
      end
    end
    return vt
  end
  
  def select_target_and_hyperspace_in_emergency!(game)
    ship = game[:ship]
    fire_delay = ready_to_fire?(game)

# FIXME: this purely distance based decision should be replaced...
    if hyperspace_necessary?(game) # double-check by distance
      @state[:last_hyperjump] = game[:sequence]
      @state[:hyperjump_count] += 1
      @controls.hyperspace
puts "HYPERSPACE EVASION NECCESARY [#{game[:sequence]}]" if $debug
      return nil
    end


    # first check for potential collisions
    colliding_objects = []
    game[:objects_by_tag].each do |tag, object|
      next if tag == :ship
      next if object[:class] == :shot && object[:tag].to_s =~ /^s/
      
      if t = Vector.will_collide?(object,ship)
#         if t <= HYPERSPACE_SAFETY_TIME+game[:latency] && @state[:strategy] != :suicide
#           if hyperspace_necessary?(game) # double-check by distance
#             @state[:last_hyperjump] = game[:sequence]
#             @state[:hyperjump_count] += 1
#             @controls.hyperspace
# puts "HYPERSPACE EVASION NECCESARY [#{game[:sequence]}] -- impact in #{t} frames" if $debug
#             return nil
#           end
#         end
      end
      
      next if object[:class] == :shot
      next if @target_blacklist.include?(tag)
      colliding_objects << { :t => t, :object => object } if t && t <= COLLISION_SAFETY_TIME+game[:latency]+fire_delay
    end

    unless colliding_objects.empty?
      o = colliding_objects.min{|o1,o2| o1[:t] <=> o2[:t]}
puts "[#{game[:sequence]}] targeting #{o[:object][:tag]} for collision prevention in #{o[:t]} frames" if $debug
      return o[:object]
    end

    return game[:asteroids].first if game[:asteroids].size == 1
    
    @chosen_target = :ufo if game[:saucer_present]
    
    selected_target = nil

    if @chosen_target # stick with chosen target
      selected_target = game[:objects_by_tag][@chosen_target]
      # give it up if it is no longer a valid target
      selected_target = nil if selected_target && @target_blacklist.include?(@chosen_target) # && @shots_at_target[@chosen_target].to_i >= (SHOTS_TILL_ANNIHILATION[selected_target[:class]]||1)
      # or if we shot at it at least once but are not ready to fire more shots
      selected_target = nil if selected_target && fire_delay > 0 && @shots_at_target[@chosen_target].to_i >= 1
puts "[#{game[:sequence]}] giving up chosen target #{@chosen_target}" if $debug && selected_target.nil?
    end
    
    return selected_target if selected_target

    object_count = game[:asteroids].size + game[:explosions].size
    possible_targets = game[:asteroids].reject{|a| @target_blacklist.include?(a[:tag]) || (object_count >= MAX_OBJECTS && (SHOTS_TILL_ANNIHILATION[a[:class]]||1) > 1)}
  
    # try to find targets that are exactly fire_delay turns "away" first
    if game[:internal_sequence] 
      t_min = 512 # some number larger than 42 turn frames + 69 shot flight frames

      turn_time = fire_delay
      possible_targets.each do |a|
        angle_index = (game[:shot_angle_index] + 3*turn_time) & 0xff
        t = Vector.shot_will_hit?(ship, angle_index, a, shot_lifetime(game, turn_time), turn_time)
        if t && t < t_min
          t_min = t
          selected_target = a.merge!(:angle_difference => turn_time)
        end
        angle_index = (game[:shot_angle_index] - 3*turn_time) & 0xff
        t = Vector.shot_will_hit?(ship, angle_index, a, shot_lifetime(game, turn_time), turn_time)
        if t && t < t_min
          t_min = t
          selected_target = a.merge!(:angle_difference => -turn_time)
        end
      end

      if selected_target
        puts "[#{game[:sequence]}] New EXACT target #{selected_target[:tag]} to be fired at at #{game[:sequence]+fire_delay} and killed at #{game[:sequence]+t_min}" if $debug
        return selected_target 
      end
    end
    
    t_min = 512 # some number larger than 42 turn frames + 69 shot flight frames

    # look n turns left
    angle_index = game[:shot_angle_index]
    turn_time = 0
    TURN_LOOKAHEAD.times do
      angle_index = (angle_index + 3) & 0xff
      turn_time += 1
      possible_targets.each do |a|
        t = Vector.shot_will_hit?(ship, angle_index, a, shot_lifetime(game, turn_time), turn_time)
        # if t && t < t_min
        #   t_min = t
        if t && ([turn_time,fire_delay].max+(t-turn_time)) < t_min
          t_min = [turn_time,fire_delay].max+(t-turn_time)
          selected_target = a.merge!(:angle_difference => turn_time)
        end
      end
    end

    # look n turns right
    angle_index = game[:shot_angle_index]
    turn_time = 0
    TURN_LOOKAHEAD.times do
      angle_index = (angle_index - 3) & 0xff
      turn_time += 1
      possible_targets.each do |a|
        t = Vector.shot_will_hit?(ship, angle_index, a, shot_lifetime(game,turn_time), turn_time)
        # if t && t < t_min
        #   t_min = t
        if t && ([turn_time,fire_delay].max+(t-turn_time)) < t_min
          t_min = [turn_time,fire_delay].max+(t-turn_time)
          selected_target = a.merge!(:angle_difference => -turn_time)
        end
      end
    end

    if selected_target
      puts "[#{game[:sequence]}] New target #{selected_target[:tag]}, to be fired at at #{game[:sequence]+[selected_target[:angle_difference].abs, fire_delay].max} and killed at #{game[:sequence]+t_min}" if $debug
      puts "[#{game[:sequence]}] no SHOT available before #{game[:sequence]+fire_delay}" if $debug && fire_delay != 0
    end
    
    return selected_target || game[:asteroids].reject{|a| @target_blacklist.include?(a[:tag])}.first || (game[:saucer_present] ? game[:saucer] : nil)
  end
  
  def target_angle(game,target_object)
    ship = game[:ship]
    angle_difference = nil
    # look 42 turns left
    angle_index = game[:shot_angle_index]
    turn_time = 0
    43.times do
      if Vector.shot_will_hit?(ship, angle_index, target_object, shot_lifetime(game), turn_time)
        angle_difference = turn_time
        break
      end
      angle_index = (angle_index + 3) & 0xff
      turn_time += 1
    end

    # look 42 turns right
    angle_index = game[:shot_angle_index]
    turn_time = 0
    42.times do
      angle_index = (angle_index - 3) & 0xff
      turn_time += 1
      if Vector.shot_will_hit?(ship, angle_index, target_object, shot_lifetime(game), turn_time)
        angle_difference = -turn_time
        break
      end
    end
    
    return angle_difference || Vector.target_angle(game, target_object)
  end
  
  def ready_to_fire?(game, spare_shots = 0)
    spare_shots = 3 if spare_shots > 3
    shot_count = game[:fired_shot_count] # @shots.size

    if shot_count < (4-spare_shots)
      Controls.fire?(@controls.status) ? 2 : 0
    else
      t = (@shots.select{|s| s[:processed]}.map{|s| s[:processed]+s[:lifetime]-game[:sequence]}.min || 1) + READY_TO_FIRE_CORRECTION
      t < 0 ? 0 : t.round
    end
  end
  
  def fire_at(game, the_target = nil)
    @controls.fire
    # if we are sure which target we are definitely going to hit -> put it
    # on the blacklist to prevent shooting at it again
    if the_target
      @shots_at_target[the_target] = @shots_at_target[the_target].to_i + 1
      @target_blacklist << the_target unless @target_blacklist.include?(the_target) || @state[:strategy] != :target
    end

    target_object = game[:objects_by_tag][the_target]
    # log shot
    @shots << { :fired    => game[:sequence],
                :ping     => game[:packet_count] & 0xff,
                :lifetime => (target_object ? target_object[:lock_on] :nil) || shot_lifetime(game), # initial assumption
                :angle    => game[:shot_angle_index],
                :target   => the_target
              }
  end
  
  def shot_lifetime(game, turntime = 0)
    game[:internal_sequence] ?  72 - ((game[:internal_sequence]+game[:shot_delay]+turntime) % 4) : 69
  end
  
end