# == Synopsis
#
# asteroids-client: Asteroids player for c't creativ'08 competition
#               by: Martin Rehfeld
#            using: (J)Ruby
#
# == Usage
#
# asteroids-client [OPTION] ... [MAME-HOST]
#
# -h, --help:
#    show help
#
# -l, --log [filename]:
#    log game progress to statistics file (name defaults to highscores.txt),
#    output will be appended
#
# -v, --visualize:
#    show game tactics visualization window (requires JRuby)
#
# -a, --autostart:
#    automatically start a new game (requires start button patched MAME)
#
# -t, --timelimit [seconds]:
#    limit game time to seconds (seconds defaults to 600). Player will
#    commit suicide as fast as possible when the time limit is reached.
#
# -p, --profile:
#    (development only) capture profiling data with ruby-prof and print a
#    report when program terminates (requires MRI and can thus not be used
#    together with --visualize)
#
# -f, --force:
#    (development only) skip check and also engange in already running game
#
# -d, --debug:
#    (development only) print debug messages
#
# MAME-HOST: The host running MAME to connect to, default: localhost.

require 'getoptlong'
require 'rdoc/usage'
require 'thread'
require 'socket' 

require 'core-extensions/string'
require 'core-extensions/array'
require 'tracker'
require 'game_state_ringbuffer'
require 'player_state'
require 'player'
require 'game_startup'

opts = GetoptLong.new(
  [ '--help',         '-h', GetoptLong::NO_ARGUMENT ],
  [ '--log',          '-l', GetoptLong::OPTIONAL_ARGUMENT ],
  [ '--visualize',    '-v', GetoptLong::NO_ARGUMENT ],
  [ '--autostart',    '-a', GetoptLong::NO_ARGUMENT ],
  [ '--timelimit',    '-t', GetoptLong::OPTIONAL_ARGUMENT ],
  [ '--profile',      '-p', GetoptLong::NO_ARGUMENT ],
  [ '--force',        '-f', GetoptLong::NO_ARGUMENT ],
  [ '--debug',        '-d', GetoptLong::NO_ARGUMENT ],
  [ '--skip-startup', '-s', GetoptLong::NO_ARGUMENT ]
)

# default options
stats_filename = nil
$visualization = false
$profiling = false
$autostart = false
$timelimit = nil
$force = false
$debug = false
$game_ready = false
$skip_startup = false

# parse options
opts.each do |opt, arg|
  case opt
  when '--help'
    RDoc::usage
  when '--log'
    stats_filename = arg == '' ? 'highscores.txt' : nil
  when '--visualize'
    $visualization = true
  when '--autostart'
    $autostart = true
  when '--timelimit'
    $timelimit = arg == '' ? 600 : arg.to_i
  when '--profile'
    $profiling = true
  when '--force'
    $force = true
  when '--debug'
    $debug = true
  when '--skip-startup'
    $skip_startup = true
  end
end

host = ARGV.length == 1 ? ARGV.shift : 'localhost'
stats_file = File.new(stats_filename,'a') if stats_filename
require 'visualizer' if $visualization
require 'ruby-prof' if $profiling

GameStartup.new unless $skip_startup

# global singleton state information
ringbuffer = GameStateRingbuffer.new(20000) # let's keep all frames of a 5 min game to avoid GC
player_state = PlayerState.new

puts "Resolving hostname #{host}"

# set up communication to MAME server
socket = UDPSocket.new
socket.connect(host, 1979) 
socket.send("ctmame@\0", 0)
socket.send("ctnameMartin Rehfeld fliegt mit Ruby\0\0", 0)

puts "Player session started."

RubyProf.start if $profiling

# Thread for receiving the vector RAM from MAME, parsing it and tagging all objects
# game states are put in a ringbuffer for later analysis
# player = nil # initialze local var for player thread
receiver = Thread.new do
  tracker = Tracker.new(player_state, stats_file)
  loop do
    begin
      frame_packet, sender_info = socket.recvfrom(1026)
      command = ringbuffer.get_controls_packet
# helps when playing asteroids.heise.de (keep latency up at 3)
#      sleep 0.004
      Controls.send_command(socket, command) if command # send immediate answer based on last frame
#      Thread.kill(player) if player && player.alive? # terminate A.I. if still processing previous frame
    rescue
      STDERR.puts "\n#{$!.message}, does host exist and is MAME up?"
      exit 1
    end
    if frame_packet.size != 1026
      if frame_packet == "game over\r\n"
        puts "Received Game over -- Final score: #{ringbuffer.get_last[:score]} -- terminating."
        break
      elsif frame_packet =~ /^busy/
        puts "MAME busy -- terminating."
        break
      else
        STDERR.puts "Received damaged packet with #{frame_packet.size} byte (expected 1026), discarding!"
        next
      end
    end
#    player = Thread.new do # run A.I. in different thread to avoid frame loss/latency
      begin
        ringbuffer << tracker.parse(frame_packet)
      rescue
        STDERR.puts "\n#{$!.message}"
        STDERR.puts $!.backtrace.join("\n")
      end
#    end
  end
  exit 0
  Thread.kill(Thread.main)
end

# attach observer(s) to ringbuffer
Player.new(ringbuffer, player_state, socket)
Visualizer.new(ringbuffer, player_state, receiver) if $visualization

# game is ready to be started (will be done by Player if autostart option given)
$game_ready = true

# only terminate receiver thread on SIGINT (^C) when profiling
Signal.trap("INT") { receiver.terminate } if $profiling

# wait until receiver is terminated by Visualizer or external signal
receiver.join if receiver.alive?

if $profiling
  profile = RubyProf.stop
  puts "Flat report"
  puts "-----------"
  RubyProf::FlatPrinter.new(profile).print(STDOUT, 0)
  puts
  puts "Call graph report"
  puts "-----------------"
  RubyProf::GraphPrinter.new(profile).print(STDOUT, 0)
end
