class Tracker
  
  # OP codes
  VCTR = 0..9
  LABS = 0xa
  HALT = 0xb
  JSRL = 0xc
  RTSL = 0xd
  JMPL = 0xe
  SVEC = 0xf
  
  # Vektor ROM objects
  SCORE_DIGIT = 0xadd..0xb63
  ZERO = 0xadd
  DIGIT_TO_VALUE = {
    0xadd => 0, # 0
    0xb2c => 0, # space
    0xb2e => 1,
    0xb32 => 2,
    0xb3a => 3,
    0xb41 => 4,
    0xb48 => 5,
    0xb4f => 6,
    0xb56 => 7,
    0xb5b => 8,
    0xb63 => 9
  }
  SPARE_LIVE = 0xa6d
  EXPLOSION = 0x880..0x8d0
    
  ASTEROID_CLASS = {
    0  => :large_asteroid,
    14 => :small_asteroid,
    15 => :medium_asteroid
  }
  
  SAUCER_CLASS = {
    14 => :small_saucer,
    15 => :large_saucer
  }
  
  TARGET_RADII = {
                      #                          sizes measured from screenshot
                      # sizes adjusted by shot misses
    :small_asteroid  => 5.5**2,  # small asteroid  (18x18)   ... 7 had misses ... 6 still had some (but not verfied in detail and aiming takes very long at r=5)
    :medium_asteroid => 14**2, # medium asteroid (36x36)
    :large_asteroid  => 30**2, # large asteroid  (72x72)
    :small_saucer    => 7**2,  # small saucer    (22x15)
    :large_saucer    => 15**2, # large saucer    (40x28)
    :ship            => 25**2, # ship            (~25x25)
  }

  COLLISION_RADII = {
                      #                          sizes measured from screenshot
                      # sizes adjusted by collisons
    :small_asteroid  => 8**2,  # small asteroid  (18x18)
    :medium_asteroid => 16**2, # medium asteroid (36x36)
    :large_asteroid  => 32**2, # large asteroid  (72x72)
    :small_saucer    => 11**2, # small saucer    (22x15)
    :large_saucer    => 20**2, # large saucer    (40x28)
    :ship            => 36**2, # ship            (~25x25)
  }
  
  SHIP_RADIUS = COLLISION_RADII[:ship]
  
  def parse(packet)
    ram = packet.unpack("v*")
    framecount = packet[-2] # incremented by server for each packet
    ping = packet[-1]       # sequence of last control command sent
    frame_lag = 0
    ping_latency = 0

    unless @sequence == 0 # we do not have any expectation on framecount/ping for the first received packet
      expected_framecount = (@framecount+1) & 0xff
      expected_ping = @packet_count & 0xff

      frame_lag = if framecount < expected_framecount # 8-bit wrap
        framecount - expected_framecount + 256
      else
        framecount - expected_framecount
      end

      ping_latency = if ping > expected_ping # 8-bit wrap
        expected_ping - ping + 256
      else 
        expected_ping - ping
      end
    end

    @framecount = framecount     # framecount is the representation of the 8-bit server side frame counter
    @sequence += (1 + frame_lag) # sequence is sort of a game time measured in frames, so correct it by frame_lag
    @packet_count += 1           # packet_count is always incremented by one and used to measure ping latency

    game = { :sequence => @sequence, :packet_count => @packet_count,
             :latency => ping_latency, :lost_frames => frame_lag,
             :ping => ping,
             :score => 0, :lives => 0,
             :asteroids => [], :shots => [], :saucer => {}, :ship => {},
             :explosions => [] }
    
    shipdetect, v1x, v1y = 0
    scoredetect = 0
    pc = 1

    while true do
      op = ram[pc] >> 12
      break if [HALT,RTSL,JMPL].include?(op)

      case op
      when LABS
        vy = ram[pc] & 0x3ff
        vx = ram[pc+1] & 0x3ff
        vs = ram[pc+1] >> 12
        scoredetect = 10000 if vx == 100 && vy == 876
        pc += 2
      when JSRL
        object_address = ram[pc] & 0xfff
        case object_address
        when 0x8f3
          c = ASTEROID_CLASS[vs]
          game[:asteroids] << { :vx => vx, :vy => vy, :type => 1, :vs => vs, :class => c, :r => TARGET_RADII[c], :r_coll => COLLISION_RADII[c] }
        when 0x8ff
          c = ASTEROID_CLASS[vs]
          game[:asteroids] << { :vx => vx, :vy => vy, :type => 2, :vs => vs, :class => c, :r => TARGET_RADII[c], :r_coll => COLLISION_RADII[c] }
        when 0x90d
          c = ASTEROID_CLASS[vs]
          game[:asteroids] << { :vx => vx, :vy => vy, :type => 3, :vs => vs, :class => c, :r => TARGET_RADII[c], :r_coll => COLLISION_RADII[c] }
        when 0x91a
          c = ASTEROID_CLASS[vs]
          game[:asteroids] << { :vx => vx, :vy => vy, :type => 4, :vs => vs, :class => c, :r => TARGET_RADII[c], :r_coll => COLLISION_RADII[c] }
        when 0x929
          game[:saucer_present] = true
          game[:saucer][:vx] = vx
          game[:saucer][:vy] = vy
          game[:saucer][:vs] = vs
          c = SAUCER_CLASS[vs]
          game[:saucer][:class] = c
          game[:saucer][:r] = TARGET_RADII[c]
          game[:saucer][:r_coll] = COLLISION_RADII[c]
        when SCORE_DIGIT
          if scoredetect != 0
            game[:score] += DIGIT_TO_VALUE[object_address] * scoredetect
            scoredetect /= 10
          end
          scoredetect = 0 if scoredetect == 1
        when SPARE_LIVE
          game[:lives] += 1
        when EXPLOSION
          game[:explosions] << { :vx => vx, :vy => vy, :class => :explosion }
        end
        pc += 1
      when SVEC
        # unused for tracking
        pc += 1
      when VCTR # might be a ship or a shot, or a ship's explosion
        dy = ram[pc] & 0x3ff
        dy = -dy if (ram[pc] & 0x400) != 0
        dx = ram[pc+1] & 0x3ff
        dx = -dx if (ram[pc+1] & 0x400) != 0
        # sf is currently unused!
        # sf = 0x200 >> ((vs + op) & 0xf)
        vz = ram[pc+1] >> 12

        if dx == 0 && dy == 0 && vz == 15 # shot
          game[:shots] << { :vx => vx, :vy => vy, :class => :shot, :r => 0 }
        elsif op == 6 && vz == 12 && dx != 0 && dy != 0 # detecting ship position and vector
          detector = shipdetect
          case detector
          when 0
            v1x = dx
            v1y = dy
            shipdetect = 1
          when 1
            game[:ship_present] = true
            game[:ship][:vx] = vx
            game[:ship][:vy] = vy
            game[:ship][:dx] = v1x - dx
            game[:ship][:dy] = v1y - dy
            game[:ship][:class] = :ship
            game[:ship][:r] = SHIP_RADIUS
            game[:ship][:r_coll] = SHIP_RADIUS
          end
        elsif shipdetect == 1
          shipdetect = 0
        end
        pc += 2
      end
    end
    
    log_statistics(game)

    return game
  end
  
  def log_statistics(game)
    if game[:sequence] <= 1 && game[:lives] > 0 && !$force
      STDERR.puts "\nGame already in progress ... refusing to enter contest!"
      exit 1
    end
    
    if @in_game
      game[:duration] = Time.now - @start_time
      if @stats_log && game[:duration].to_i > 0 && game[:duration].to_i % 60 == 0 && @last_logged_intermediate_result != game[:duration].to_i
        @stats_log.print "  Intermediate result after #{game[:duration]} seconds: cruising at #{(game[:score] + @score_correction) / game[:duration]} points/s, Score: #{game[:score] + @score_correction}\n"
        @stats_log.flush
        @last_logged_intermediate_result = game[:duration].to_i
puts "Intermediate result after #{game[:duration]} seconds: cruising at #{(game[:score] + @score_correction) / game[:duration]} points/s, Score: #{game[:score] + @score_correction}"
      end
    end
    
    new_lives = game[:lives]
    if @lives == 0 && new_lives > 0 # game started
      @start_time = Time.now
      @in_game = true
      if @stats_log
        @stats_log.print "Game started at #{@start_time}\n"
        @stats_log.flush
puts "Game started at #{@start_time} (sequence = #{game[:sequence]})"
      end
      @score_correction = 0
      @score = 0
    elsif @lives > 0 && new_lives == 0 # game ended
      @in_game = false
      if @stats_log
        @stats_log.print "Game ended after #{game[:duration]} seconds at #{Time.now} with #{(game[:score] + @score_correction) / game[:duration]} points/s, Score: #{game[:score] + @score_correction}\n"
        @stats_log.flush
puts "Game ended after #{game[:duration]} seconds at #{Time.now} with #{(game[:score] + @score_correction) / game[:duration]} points/s, Score: #{game[:score] + @score_correction}"
      end
    elsif @lives > new_lives
      if @stats_log
        @stats_log.print "  Lost life [#{@sequence}] after #{game[:duration]} seconds, cruising at #{(game[:score] + @score_correction) / game[:duration]} points/s, Score: #{game[:score] + @score_correction}\n"
        @stats_log.flush
puts "Lost life [#{@sequence}] after #{game[:duration]} seconds, cruising at #{(game[:score] + @score_correction) / game[:duration]} points/s, Score: #{game[:score] + @score_correction}"

        if @state[:last_hyperjump].nil? || (@state[:last_hyperjump] < game[:sequence]-49 && @state[:strategy] != :suicide)
        @stats_log.print "  Last hyperjump occurred more than 49 frame earlier, so we probably died by collision, here is the current game dump:\n#{game.inspect}\n"
        @stats_log.flush
puts "Last hyperjump occurred more than 49 frames earlier, so we probably died by collision, here is the current game dump:\n#{game.inspect}\n"
        end
      end
    end
    @lives = new_lives

    game[:enabled] = @in_game

    if game[:score] < @score
      @score_correction += 100000
    end
    
    @score = game[:score]
    game[:score] += @score_correction
  end
  
  def initialize(state = {}, stats_log = nil)
    @sequence = 0
    @packet_count = 0
    @stats_log = stats_log
    @lives = 0
    @score_correction = 0
    @score = 0
    @in_game = false
    @state = state
  end
    
end
