//
//  APTracker.m
//  APlayer
//
//  Created by Holger Sadewasser on 4/19/08.
//  Copyright 2008. All rights reserved.
//

#import "APTracker.h"
#import "APObject.h"
#import "parameter.h"
#import "tools.h"
#ifdef WITH_WINDOW
#import "APView.h"
extern APView *gView;
#endif


// methods that are used internally by this class, but should not be needed externally
// are defined here to keep them "private". Anyone who knows about them can still call
// them of course, but by putting the prototypes here, it emphasizes that they are only
// meant for internal use.
@interface APTracker(_private_methods)

-(APObject *)_identifyObjectWithType:(APObjectTypes_t)iType size:(int)iSize posX:(int)iPosX posY:(int)iPosY frames:(uint8_t)iFrames ping:(uint8_t)iPing;

@end


@implementation APTracker

// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Initializers/Clean up
// -------------------------------------------------------------------------------------

-(id)initWithIOChannel:(APIOChannel *)iIOChannel {
  int i;
  
  self = [super init];
  if (self != nil) {
    mIOChannel = iIOChannel;
    [mIOChannel retain];
//    [mIOChannel setDataCallbackObject:self];
    [mIOChannel setWriteCallbackObject:self];
    
    mFrameNo = 0;
    mTotalFrames = 0;
    mLostFrames = 0;
    mFlgFrameNoSet = NO;
    
    mFrames = [[NSMutableData alloc] init];
    mFlgRecordFrames = NO;
    mRecordedFrames = 0;
    
    mLastPing = 0;
    
    for ( i = 0; i < 256; i++ ) {
      mKeys[i] = 0;
      mTimeToLive[i] = -1;
    }
    
    mShip = [[APShip alloc] initWithKeysBuffer:mKeys];
    mSaucer = [[APSaucer alloc] init];    
    mFlgShipInitialized = YES;
    mFlgSaucerInitialized = YES;
    mFlgSynchronize = NO;
    mSyncID = 0;
    
    mNumAsteroids = 0;
    mNumShots = 0;
    mNumOwnShots = 0;
    
//    mSnapshots = [[NSMutableArray alloc] initWithCapacity:1000];
  }
  return self;
}


-(void)dealloc { 
//  NSLog(@"snapshots: %d", [mSnapshots count]);
//  [mSnapshots release];
  [mObjects release];
  [mSaucer release];
  [mShip release];
  [mFrames release];
//  [mIOChannel setDataCallbackObject:nil];
  [mIOChannel setWriteCallbackObject:nil];
  [mIOChannel release];
  [super dealloc];
}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Instance Methods
// -------------------------------------------------------------------------------------

-(void)didSendKeys:(uint8_t)iKeys ping:(uint8_t)iPing timeToLive:(int)iTimeToLive {
 
  uint8_t count = (uint8_t)(iPing - mLastPing);
  
  while ( count > 1 ) {
    ++mLastPing;
    mKeys[mLastPing] = 0;
    mTimeToLive[mLastPing] = -1;
    --count;
  }
  
  mLastPing = iPing;
  mKeys[mLastPing] = iKeys;
  mTimeToLive[mLastPing] = iTimeToLive;
  
//  NSLog(@"did send %#x at ping %d", (int)iKeys, (int)iPing);
}


-(NSArray *)interpretScreen:(NSData *)iPacket {
  
	unsigned short * vector_ram = (unsigned short *)[iPacket bytes];
  unsigned short vr, vr1;
	int dx, dy, sf, vx, vy, vz, vs;
	int v1x = 0;
	int v1y = 0;
	int shipdetect = 0;
  NSMutableArray * newObjects = [[NSMutableArray alloc] init];
  APObject * object;
  int count, i;
  BOOL flgDone = NO;
  BOOL flgFound;
  NSEnumerator * enumerator;
  
  uint8_t frameno = ((uint8_t *)vector_ram)[1024];
  uint8_t ping = ((uint8_t *)vector_ram)[1025];

  if ( mFlgFrameNoSet == NO ) {
//    NSLog(@"initial frame %d", (int)frameno);
//    NSLog(@"initial ping %d  last ping %d", (int)ping, (int)mLastPing);
    mFlgFrameNoSet = YES;
    mFrameNo = frameno - 1;
  }
  
  uint8_t frames = frameno - mFrameNo;
  
  mTotalFrames += frames;
  mLostFrames = frames - 1;
  
//  if ( mLostFrames > 0 ) {
//    NSLog(@"lost frames: %d", mLostFrames);
//  }
//  if ( ping != mLastPing ) {
//    NSLog(@"latency %d", (int)(ping-mLastPing));
//  }
  int numAsteroids = 0;
  int numShots = 0;
  int numOwnShots = 0;
  int flgShipPresent = NO;
  int flgSaucerPresent = NO;
  
  
	if (((unsigned char *)vector_ram)[1] != 0xe0 && ((unsigned char *)vector_ram)[1] != 0xe2)
		return mObjects; // sollte nicht vorkommen; erster Befehl ist immer ein JMPL
  
	int pc = 1;
	while ( flgDone == NO && pc < 512 )
	{
    vr  = EndianU16_LtoN(vector_ram[pc]);
    vr1 = EndianU16_LtoN(vector_ram[pc+1]);
    
		int op = vr >> 12;
		switch (op)
		{
      case 0xa: // LABS
        vy = vr & 0x3ff;
        vx = vr1 & 0x3ff;
        vs = vr1 >> 12;
        break;
      case 0xb: // HALT
        flgDone = YES;
        break;
      case 0xc: // JSRL
        switch (vr & 0xfff)
        {
          case 0x8f3:
            [newObjects addObject:[self _identifyObjectWithType:cAsteroid1 size:vs posX:vx posY:vy frames:frames ping:ping]];
            ++numAsteroids;
            break;
          case 0x8ff:
            [newObjects addObject:[self _identifyObjectWithType:cAsteroid2 size:vs posX:vx posY:vy frames:frames ping:ping]];
            ++numAsteroids;
            break;
          case 0x90d:
            [newObjects addObject:[self _identifyObjectWithType:cAsteroid3 size:vs posX:vx posY:vy frames:frames ping:ping]];
            ++numAsteroids;
            break;
          case 0x91a:
            [newObjects addObject:[self _identifyObjectWithType:cAsteroid4 size:vs posX:vx posY:vy frames:frames ping:ping]];
            ++numAsteroids;
            break;
          case 0x929:
            if ( mFlgSaucerPresent == NO )
              [mSaucer setSize:vs posX:vx posY:vy];
            else
              [mSaucer updateWithPosX:vx posY:vy frames:frames];
            [newObjects addObject:mSaucer];
            flgSaucerPresent = YES;
            mFlgSaucerInitialized = NO;
            break;
        }  
        break;
      case 0xd: // RTSL
        flgDone = YES;
        break;
      case 0xe: // JMPL
        /*
         pc = vr & 0xfff;
         break;
         */
        flgDone = YES;
        break;
      case 0xf: // SVEC
        /*
         dy = vr & 0x300;
         if ((vr & 0x400) != 0)
         dy = -dy;
         dx = (vr & 3) << 8;
         if ((vr & 4) != 0)
         dx = -dx;
         sf = (((vr & 8) >> 2) | ((vr & 0x800) >> 11)) + 2;
         vz = (vr & 0xf0) >> 4;
         */
        break;
      default:
        dy = vr & 0x3ff;
        if ( (vr & 0x400) != 0 )
          dy = -dy;
        dx = (vr1 & 0x3ff);
        if ((vr1 & 0x400) != 0)
          dx = -dx;
        sf = op;
        vz = vr1 >> 12;
        if (dx == 0 && dy == 0 && vz == 15) {
          [newObjects addObject:[self _identifyObjectWithType:cShot size:0 posX:vx posY:vy frames:frames ping:ping]];
          ++numShots;
        } else if (dx == 0 && dy == 0 && vz != 0 ) {
          NSLog(@"shot with vz = %d", (int)vz);
        }
        if (op == 6 && vz == 12 && dx != 0 && dy != 0)
        {
            switch (shipdetect)
          {
            case 0:
              v1x = dx;
              v1y = dy;
              ++shipdetect;
              break;
            case 1:
              if ( mFlgShipPresent == NO )
                [mShip setPosX:vx posY:vy viewX:(v1x-dx) viewY:(v1y-dy)];
              else
                [mShip updateWithPosX:vx posY:vy viewX:(v1x-dx) viewY:(v1y-dy) frame:frameno frames:frames ping:ping lastPing:mLastPing];
//              NSLog(@"Ship view %d %d at ping: %d\n", (v1x-dx), (v1y-dy), (int)ping);
              [newObjects addObject:mShip];
              flgShipPresent = YES;
              mFlgShipInitialized = NO;
              ++shipdetect;
              break;
          }
        }
          else if (shipdetect == 1)
            shipdetect = 0;
          
          break;
		}
		if (op <= 0xa)
			++pc;
		if (op != 0xe) // JMPL
			++pc;
	}   
  
//- Begin Records shot angles ---------------------
//  unsigned int count = [newObjects count];
//  unsigned int index;
//  APObject * object;
//  for ( index = 0; index < count; index++ ) {
//    
//    object = [newObjects objectAtIndex:index];
//    
//    if ( object->mType == cShot ) {
////      NSLog(@"tf: %d", object->mTotalFrames);
//      if ( object->mTotalFrames == 1 ) {
////        NSLog(@"ox: %lf\toy: %lf\tsx: %lf\tsy: %lf", object->mInitPosX, object->mInitPosY, mShip->mPosX, mShip->mPosY);
//        double dx = object->mInitPosX - mShip->mPosX;
//        double dy = object->mInitPosY - mShip->mPosY;
////        NSLog(@"dx: %lf\tdy: %lf\tdist: %lf\tvx: %lf\tvy: %lf", dx, dy, dx*dx + dy*dy, mShip->mViewX, mShip->mViewY);
//        if ( dx*dx + dy*dy < 485.0 ) {
//          object->mShipViewX = mShip->mViewX;
//          object->mShipViewY = mShip->mViewY;
//          object->mPing = ping;
//          object->mAngleByteTmp = mAngleByte;
//          object->mFlgRecording = YES;
//          ++numShots;
//        }
//      } else if ( object->mShipViewX != 0.0 || object->mShipViewY != 0.0 ) {
//        ++numShots;
//      }
//    }
//  }
//- End Records shot angles ---------------------
  
  if ( mFlgRecordFrames == YES && mRecordedFrames <= 100 ) {
    int i;
    for ( i = pc+1; i < 512; i++ ) {
      vector_ram[i] = 0;
    }
    ((unsigned char *)vector_ram)[1] = 0xe0;
    [mFrames appendBytes:vector_ram length:1024];
    mRecordedFrames++;
    if ( mRecordedFrames == 100 ) {
      [mFrames writeToFile:@"/Users/holger/MAME/frames.dat" atomically:NO];
      mFlgRecordFrames = NO;
    }
  }
  
  count = [newObjects count];
  for ( i = 0; i < count; i++ ) {
    object = [newObjects objectAtIndex:i];
    if ( [object type] == cShot ) {
      if ( [object totalFrames] == 1 ) {
        dx = [mShip posXInt] - [object posXInt];
        dy = [mShip posYInt] - [object posYInt];
        if ( dx*dx + dy*dy <= 441.0 ) {
          [object setOwnShot:YES];
          ++numOwnShots;
        } else
          [object setTimeToLive:72];
      } else if ( [object isOwnShot] == YES )
        ++numOwnShots;
    }
  }
  
//  if ( mNumOwnShots != numOwnShots ) {
//    NSLog(@"own shots: %d", numOwnShots);
//  }
  if ( mFlgSynchronize == YES && mSyncID != 0 ) {
    flgFound = NO;
    enumerator = [newObjects objectEnumerator];
    while( (object = [enumerator nextObject]) && flgFound == NO ) {
      if ( [object type] == cShot && [object ID] == mSyncID ) {
        if ( [object calculatingHeading] == NO ) {
//          NSLog(@"hx: %f  hy; %f", [object headingX], [object headingY]);
          [mShip synchronizeAngleByteFromHeadingX:(int)([object headingX]*8.0) headingY:(int)([object headingY]*8.0)];
          mFlgSynchronize = NO;
          mSyncID = 0;
        }
        flgFound = YES;
      }
    }
    if ( flgFound == NO ) {
      mFlgSynchronize = NO;
      mSyncID = 0;
    }
  }
  
  mNumAsteroids = numAsteroids;
  mNumShots = numShots;
  mNumOwnShots = numOwnShots;
  mFlgShipPresent = flgShipPresent;
  mFlgSaucerPresent = flgSaucerPresent;  
  mFrameNo = frameno;
  if ( mFlgSaucerPresent == NO && mFlgSaucerInitialized == NO ) {
    [mSaucer init];
    mFlgSaucerInitialized = YES;
  }
  if ( mFlgShipPresent == NO && mFlgShipInitialized == NO ) {
    [mShip init];
    mFlgShipInitialized = YES;
  }
  
  //  NSLog(@"asteroids: %d\n", mNumAsteroids);
  //  NSLog(@"shots: %d\n", mNumShots);
  //  NSLog(@"ship: %d\n", flgShipPresent);
  //  NSLog(@"saucer: %d\n", flgSaucerPresent);

//  int index;
//  APObject * object;
//  int count = [mObjects count];
//  for ( index = 0; index < count; index++ ) {
//    object = [mObjects objectAtIndex:index];
//    NSLog(@"retain count: %u  total frames: %u", [object retainCount], object->mTotalFrames);
//  }
  [mObjects release];
  mObjects = newObjects;
  
#ifdef WITH_WINDOW
  [gView setObjects:mObjects];
  [gView setNeedsDisplay:YES];
#endif
  
  return mObjects;
}


-(void)shotAvailability:(int [])xBuffer {

  NSMutableArray * ownShots = [[NSMutableArray alloc] initWithCapacity:4];
  NSEnumerator * enumerator;
  APObject * object;
  int count = 0;
  
  enumerator = [mObjects objectEnumerator];
  while( (object = [enumerator nextObject]) && count < 4 ) {
    if ( [object type] == cShot && [object isOwnShot] == YES ) {
      [ownShots addObject:object];
      ++count;
    }
  }
  
  [ownShots sortUsingSelector:@selector(compareTimeToLiveDescendingToObject:)];
  
  count = 3;
  enumerator = [ownShots objectEnumerator];
  while( (object = [enumerator nextObject]) ) {
    xBuffer[count] = [object timeToLive];
  }
  while( count >= 0 ) {
    xBuffer[count] = 0;
    --count;
  }

  [ownShots release];
}


-(uint8_t)lastPing {
  return mLastPing;
}


-(unsigned int)totalFrames {
  return mTotalFrames;
}

-(unsigned int)lostFrames {
  return mLostFrames;
}

-(int)numberOfAsteroids {
  return mNumAsteroids;
}

-(int)numberOfShots {
  return mNumShots;
}

-(int)numberOfOwnShots {
  return mNumOwnShots;
}

-(BOOL)shipPresent {
  return mFlgShipPresent;
}

-(BOOL)saucerPresent {
  return mFlgSaucerPresent;
}

-(APShip *)ship {
  return mShip;
}

-(APSaucer *)saucer {
  return mSaucer;
}

-(NSMutableArray *)objects {
  return mObjects;
}

-(void)setSynchronize:(BOOL)iFlgSynchronize {
  mFlgSynchronize = iFlgSynchronize;
}

//-(void)createSnapshot {
//  [mSnapshots addObject:[mObjects copy]];
//}


// -------------------------------------------------------------------------------------
#pragma mark -
#pragma mark Private Methods
// -------------------------------------------------------------------------------------

-(APObject *)_identifyObjectWithType:(APObjectTypes_t)iType size:(int)iSize posX:(int)iPosX posY:(int)iPosY frames:(uint8_t)iFrames ping:(uint8_t)iPing {
  
  unsigned int index;
  unsigned int count = [mObjects count];
  APObject * object;
  double distX, distY, dist;
  double distXa, distYa;
  double divX, divY, div;
  int timeToLive;
    
  double singleFrameDist = 1e10;
  double singleFrameIndex = -1;
  APObject * singleFrameObject = nil;

  double multiFrameDist = 1e10;
  double multiFrameDiv = 1e10;
  double multiFrameIndex = -1;
  APObject * multiFrameObject = nil;
  
  for ( index = 0; index < count; index++ ) {
    
    object = [mObjects objectAtIndex:index];
    
    if ( object->mType == iType && object->mSize == iSize ) {
      distX = distXdouble( object->mPosX, (double)iPosX );
      distY = distYdouble( object->mPosY, (double)iPosY );      
      distXa = fabs(distX);
      distYa = fabs(distY);
      dist = distX*distX + distY*distY;
//      if ( dist < TRACKING_POS_RANGE  ) {
      if ( (iType == cShot && distXa < TRACKING_POS_RANGE+1.0 && distYa < TRACKING_POS_RANGE+1.0) ||
           (distXa < TRACKING_POS_RANGE && distYa < TRACKING_POS_RANGE) ) {
        if ( object->mTotalFrames == 1 ) {
          if ( dist < singleFrameDist ) {
            singleFrameDist = dist;
            singleFrameObject = object;
            singleFrameIndex = index;
          }
        } else {
          divX = fabs(distX - object->mHeadingX);
          divY = fabs(distY - object->mHeadingY);
          div = divX*divX + divY*divY;
//          if ( div < TRACKING_HEADING_DIV) {
          if ( divX < TRACKING_HEADING_DIV && divY < TRACKING_HEADING_DIV ) {
            if ( div < multiFrameDiv ) {
              multiFrameDist = dist;
              multiFrameDiv = div;
              multiFrameObject = object;
              multiFrameIndex = index;
            }
          }
        }
      }
    }
  }
  
  if ( multiFrameObject != nil ) {
    object = multiFrameObject;
    index = multiFrameIndex;
    //    NSLog(@"Object identified 2 type: %d size: %d x: %lf -> %d y: %lf -> %d dist: %lf hx: %lf hy: %lf div: %lf f: %d\n",
    //          iType, iSize, object->mPosX, iPosX, object->mPosY, iPosY, multiFrameDist, object->mHeadingX, object->mHeadingY, multiFrameDiv, object->mTotalFrames);
  } else if ( singleFrameObject != nil ) {
    object = singleFrameObject;
    index = singleFrameIndex;
    //    NSLog(@"Object identified 1 type: %d size: %d x: %lf -> %d y: %lf -> %d dist: %lf hx: %lf hy: %lf f: %d\n",
    //          iType, iSize, object->mPosX, iPosX, object->mPosY, iPosY, singleFrameDist, object->mHeadingX, object->mHeadingY, object->mTotalFrames);
  } else {
    object = nil;
  }
  
  if ( object != nil ) {
    [object retain];
    [mObjects removeObjectAtIndex:index];
    [object updateWithPosX:iPosX posY:iPosY frames:iFrames];
    
    //    NSLog(@"Object type: %d size: %d x: %lf y: %lf hx: %lf hy: %lf f: %d\n",
    //          object->mType, object->mSize, object->mPosX, object->mPosY, object->mHeadingX, object->mHeadingY, object->mTotalFrames);

    return [object autorelease];
  } else {
//    NSLog(@"New object type: %d size: %d x: %d y: %d\n", iType, iSize, iPosX, iPosY);
//    if ( iType == cShot ) NSLog(@"Angle byte: %d  new shot at x: %d  y: %d  ping: %d\n", (int)(mShip->mAngleByte), iPosX, iPosY, (int)iPing);
    object = [[APObject alloc] initWithType:iType size:iSize posX:iPosX posY:iPosY];
    if ( iType == cShot ) {
      // this works only with no latency
      if ( mLastPing != iPing )
        timeToLive = mTimeToLive[(uint8_t)(mLastPing-1)];
      else
        timeToLive = mTimeToLive[(uint8_t)(iPing-1)];
      if ( timeToLive >= 0 && timeToLive < 72 ) {
        [object setTimeToLive:timeToLive];
      }
//      NSLog(@"ttl: %d", timeToLive);
      if ( mFlgSynchronize == YES && mSyncID == 0 ) {
        mSyncID = [object ID];
      }
    }
    return [object autorelease];
  }
}


@end
