package de.fhr.asteroids;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;

import javax.swing.SwingUtilities;

/**
 * A display to show the running asteroid game on the client.
 * @author Florian Lutz
 * @version 1.1
 */
final class Display extends java.awt.Frame implements KeyListener {

  /**
   * The buffer image for double buffered drawing.
   */
  private Image buffer;

  /**
   * The graphical context to use for drawing.
   */
  private final FlipGraphics fg;

  /**
   * If the display is freezed and not updated.
   */
  private boolean freeze;

  /**
   * The instance of the playing bot.
   */
  private final Drawable player;

  /**
   * The state of the game to be drawn.
   */
  private final Drawable state;

  /**
   * If the abort key has been pressed on the display.
   */
  boolean aborted;

  /**
   * Creates a new display for the running asteroids game.
   * @param state the state of the game to draw
   * @param player the playing bot to draw
   */
  Display(final Drawable state, final Drawable player) {
    super();
    this.state = state;
    this.player = player;
    fg = new FlipGraphics(Tables.EXTENT_X, Tables.EXTENT_Y);
    addKeyListener(this);
    setSize(Tables.EXTENT_X, Tables.EXTENT_Y);
    setUndecorated(true);
    setBackground(Color.BLACK);
    setTitle("Asteroids");
    System.out.println("Showing display");
    setVisible(true);
  }

  /**
   * {@inheritDoc}
   * @see java.awt.event.KeyListener#keyPressed(java.awt.event.KeyEvent)
   */
  public void keyPressed(final KeyEvent e) {
    final int kc = e.getKeyCode();
    if (kc == 27) {
      aborted = true;
    } else if (kc == 32) {
      togglefreeze();
    }
  }

  /**
   * {@inheritDoc}
   * @see java.awt.event.KeyListener#keyReleased(java.awt.event.KeyEvent)
   */
  public void keyReleased(final KeyEvent e) {
    // not needed
  }

  /**
   * {@inheritDoc}
   * @see java.awt.event.KeyListener#keyTyped(java.awt.event.KeyEvent)
   */
  public void keyTyped(final KeyEvent e) {
    // not needed
  }

  /**
   * {@inheritDoc}
   * @see java.awt.Container#paint(java.awt.Graphics)
   */
  @Override
  public void paint(final Graphics g) {
    if (buffer == null) {
      buffer = createImage(Tables.EXTENT_X, Tables.EXTENT_Y);
    }
    if (!freeze) {
      final Graphics bg = buffer.getGraphics();
      bg.setColor(Color.BLACK);
      bg.fillRect(0, 0, Tables.EXTENT_X, Tables.EXTENT_Y);
      fg.setGraphics(bg);
      state.draw(fg);
      player.draw(fg);
      bg.setColor(Color.WHITE);
      fg.drawString("Press ESC to exit, SPACE to freeze display", 10, 10);
    }
    g.drawImage(buffer, 0, 0, this);
  }

  /**
   * {@inheritDoc}
   * @see java.awt.Container#update(java.awt.Graphics)
   */
  @Override
  public void update(final Graphics g) {
    paint(g);
  }

  /**
   * Toggles the freezing of the display.
   */
  private void togglefreeze() {
    if (freeze) {
      freeze = false;
    } else {
      freeze = true;
    }
  }

  /**
   * Serialisierungs-ID.
   */
  private static final long serialVersionUID = 1L;

  /**
   * Asynchronously disposes a display instance.
   * @param display the display instance to dispose
   */
  static void dispose(final Display display) {
    if (display == null) {
      return;
    }
    SwingUtilities.invokeLater(new Runnable() {
      @Override
      public void run() {
        display.dispose();
      }
    });
  }
}
