package de.fhr.asteroids;

/**
 * Superclass for all objects of the asteroids game.
 * @author Florian Lutz
 * @version 1.1
 */
abstract class GameObject implements Drawable {

  /**
   * Saved x-values of the objects positions.
   */
  private final int[] savepx = new int[20];

  /**
   * Saved y-values of the objects positions.
   */
  private final int[] savepy = new int[20];

  /**
   * The current x-value of the objects position.
   */
  int px;

  /**
   * The current y-value of the objects position.
   */
  int py;

  /**
   * The speed of the object.
   */
  double speed;

  /**
   * If the object has been tracked an updated.
   */
  boolean updated;

  /**
   * The x-value of the objects velocity vector.
   */
  double vx;

  /**
   * The y-value of the objects velocity vector.
   */
  double vy;

  /**
   * Creates a new game object with the specified position.
   * @param px the x-value of the object position
   * @param py the y-value of the object position
   */
  GameObject(final int px, final int py) {
    this.px = px;
    this.py = py;
    updated = true;
  }

  /**
   * Calculates the time of the minumum distance to another object.
   * @param opx the x-value of the other objects position
   * @param opy the y-value of the other objects position
   * @param ovx the x-value of the other objects velocity vector
   * @param ovy the y-value of the other objects velocity vector
   * @return the time when the minimum distance is will be reached
   */
  private double mintime(final int opx, final int opy, final double ovx,
      final double ovy) {
    double vx1 = vx;
    double vy1 = vy;
    if (this instanceof Shot) {
      vx1 = Util.sspx(((Shot)this).ab, 0);
      vy1 = Util.sspy(((Shot)this).ab, 0);
    }
    return (-((py - opy) * (vy1 - ovy)) - ((px - opx) * (vx1 - ovx)))
        / (((vx1 - ovx) * (vx1 - ovx)) + ((vy1 - ovy) * (vy1 - ovy)));
  }

  /**
   * Recalculates the spees of the object after an update.
   */
  private void recalc() {
    double sumx = px - savepx[0];
    double sumy = py - savepy[0];
    int n = 1;
    for (int i = 0; i < 19; i++) {
      if (savepx[i + 1] > 0 && savepy[i + 1] > 0) {
        final int ax = savepx[i] - savepx[i + 1];
        final int ay = savepy[i] - savepy[i + 1];
        sumx += ax;
        sumy += ay;
        n++;
      }
    }
    vx = (vx + sumx) / (n + 1);
    vy = (vy + sumy) / (n + 1);
    speed = Util.vlen(vx, vy);
  }

  /**
   * Saves the old objects position befor an update.
   * @param opx the x-value of the old position
   * @param opy the y-value of the old position
   */
  private void save(final int opx, final int opy) {
    for (int i = 19; i >= 1; i--) {
      savepx[i] = savepx[i - 1];
      savepy[i] = savepy[i - 1];
    }
    savepx[0] = opx;
    savepy[0] = opy;
  }

  /**
   * Warps the saved positions around the field.
   * @param npx the x-value of the relation point
   * @param npy the y-value of the relation point
   */
  private void warp(final int npx, final int npy) {
    for (int i = 0; i < 20; i++) {
      savepx[i] = Util.warpx(savepx[i], npx);
      savepy[i] = Util.warpy(savepy[i], npy);
    }
  }

  /**
   * Returns the distance to another object.
   * @param other the other object
   * @return the distance
   */
  final double distance(final GameObject other) {
    return Util.distance(px, py, other.px, other.py);
  }

  /**
   * Returns the distance to another object after the specified time.
   * @param other the other object
   * @param t the time
   * @return the distance
   */
  final double distance(final GameObject other, final double t) {
    return Util.distance(px + t * vx, py + t * vy, other.px + t * other.vx,
        other.py + t * other.vy);
  }

  /**
   * Returns the distance to another object after the specified time, warping
   * around the field.
   * @param x the x-value of the other objects position
   * @param y the y-value of the other objects position
   * @return the distance
   */
  final double distance2(final double x, final double y) {
    return Util.distance2(px, py, x, y);
  }

  /**
   * Returns the distance to another object, warping around the field.
   * @param other the other object
   * @return the distance
   */
  final double distance2(final GameObject other) {
    return Util.distance2(px, py, other.px, other.py);
  }

  /**
   * Returns the distance to another object after the specified time, warping
   * around the field.
   * @param other the other object
   * @param t the time
   * @return the distance
   */
  final double distance2(final GameObject other, final double t) {
    return Util.distance2(px + t * vx, py + t * vy, other.px + t * other.vx,
        other.py + t * other.vy);
  }

  /**
   * Draws the velocity vector of this game object.
   * @param g the graphical context to draw on
   */
  final void drawSpeed(final FlipGraphics g) {
    g.drawLine(px, py, px + (int)(10.0 * vx), py + (int)(10.0 * vy));
  }

  /**
   * Calulates the minimum distance to another object.
   * @param other the other object
   * @return the minimum distance
   */
  final double mindist(final GameObject other) {
    return distance2(other, mintime(other));
  }

  /**
   * Calculates the time of the minumum distance to another object.
   * @param other the other object
   * @return the time when the minimum distance is will be reached
   */
  final double mintime(final GameObject other) {
    double ovx = other.vx;
    double ovy = other.vy;
    if (other instanceof Shot) {
      ovx = Util.sspx(((Shot)other).ab, 0);
      ovy = Util.sspy(((Shot)other).ab, 0);
    }
    double time = mintime(other.px, other.py, ovx, ovy);
    if (time < 0) {
      final int opx = Util.warpx(other.px, px);
      final int opy = Util.warpy(other.py, py);
      time = mintime(opx, opy, ovx, ovy);
    }
    return time;
  }

  /**
   * Returns the radius of this game object.
   * @return the objects radius
   */
  abstract int radius();

  /**
   * Updates the position of this object to calculate its velocity vector.
   * @param npx the x-value of the new position
   * @param npy the y-value of the new position
   */
  final void update(final int npx, final int npy) {
    updated = true;
    save(px, py);
    if (Math.abs(px - npx) > 200 || Math.abs(py - npy) > 200) {
      warp(npx, npy);
    }
    px = npx;
    py = npy;
    recalc();
  }
}
