package de.fhr.asteroids;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

/**
 * The bot implementaion playing the asteroids game.
 * @author Florian Lutz
 * @version 1.1
 */
final class Player implements Drawable {

  /**
   * Comparator for the distance from a shot to targets.
   */
  private static class ShotComp implements Comparator<Target> {

    /**
     * The Shot to compare with.
     */
    private Shot shot;

    /**
     * Creates a new comparator object.
     */
    ShotComp() {
      super();
    }

    /**
     * {@inheritDoc}
     * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
     */
    @Override
    public int compare(final Target o1, final Target o2) {
      if (shot != null) {
        return Double.compare(shot.distance2(o1), shot.distance2(o2));
      }
      return 0;
    }

    /**
     * Sets the shot to compate with.
     * @param shot the shot
     */
    void setShot(final Shot shot) {
      this.shot = shot;
    }
  }

  /**
   * The control to operate the game.
   */
  private final Control control;

  /**
   * The length of the firesequence.
   */
  private int fireseqlen;

  /**
   * Counter for the fire sequence.
   */
  private int firesequence;

  /**
   * If left turn has ben sent.
   */
  private boolean sentleft;

  /**
   * If right turn has ben sent.
   */
  private boolean sentright;

  /**
   * The players ship.
   */
  private Ship ship;

  /**
   * The comparator for targets with shots.
   */
  private final ShotComp shotcomp;

  /**
   * The lists of targets that would be hit by shots.
   */
  private final List<List<Target>> shothits;

  /**
   * The state of the game.
   */
  private final GameState state;

  /**
   * The current target object.
   */
  private Target target;

  /**
   * The targets on the field.
   */
  private final List<Target> targets;

  /**
   * An internal test shot to calculate the time to shoot.
   */
  private final Shot testshot;

  /**
   * Creates a new bot for the given state and control.
   * @param state the game state that contains the necessary information
   * @param control the control to operate the game
   */
  Player(final GameState state, final Control control) {
    super();
    this.state = state;
    this.control = control;
    targets = new ArrayList<Target>();
    testshot = Shot.create(0, 0);
    shothits = new ArrayList<List<Target>>();
    for (int i = 0; i < 8; i++) {
      shothits.add(i, new ArrayList<Target>());
    }
    shotcomp = new ShotComp();
  }

  /**
   * {@inheritDoc}
   * @see de.fhr.asteroids.Drawable#draw(de.fhr.asteroids.FlipGraphics)
   */
  public synchronized void draw(final FlipGraphics g) {
    if (ship != null && target != null && target.aimpoint.ab >= 0) {
      g.setColor(Color.PINK);
      final double dx = Util.vectorx(Tables.ANGLE[target.aimpoint.ab], 100);
      final double dy = Util.vectory(Tables.ANGLE[target.aimpoint.ab], 100);
      g.drawRay(ship.px, ship.py, dx, dy);
      g.drawString("" + target.aimpoint.ab, ship.px, ship.py - ship.radius()
          - 24);
    }
    g.setColor(Color.DARK_GRAY);
    for (Target t : targets) {
      if (t.willcollide) {
        final int x1 = t.px + (int)Util.vxscale(t.vy, t.vx, t.radius() + 2);
        final int y1 = t.py - (int)Util.vyscale(t.vy, t.vx, t.radius() + 2);
        final int x2 = t.px - (int)Util.vxscale(t.vy, t.vx, t.radius() + 2);
        final int y2 = t.py + (int)Util.vyscale(t.vy, t.vx, t.radius() + 2);
        g.drawRay(x1, y1, t.vx, t.vy);
        g.drawRay(x2, y2, t.vx, t.vy);
      }
    }
  }

  /**
   * Calculates paramters on a target.
   * @param tg the target to calculate
   */
  private void calcTarget(final Target tg) {
    tg.willbehit = false;
    tg.willcollide = false;
    if (tg.hasbeenshot > 0) {
      tg.hasbeenshot--;
    }

    if (ship == null) {
      tg.shipdist = 0;
      tg.aimpoint.take(tg);
      tg.firepoint.take(tg);
      return;
    }

    tg.shipdist = ship.distance2(tg);
    tg.aimpoint.take(ship.aimpoint(tg));
    tg.firepoint.take(ship.firepoint(tg.aimpoint));

    final int secdist = tg.radius() + ship.radius();
    if (tg.mindist(ship) <= secdist) {
      tg.willcollide = true;
      if (tg.shipdist < secdist + (int)(tg.speed * 2)) {
        control.hyperspace(true);
      }
    }

    for (Shot s : state.shots) {
      final double t = s.mintime(tg);
      if (!s.hostile && s.distance(tg, t) <= tg.radius() + 5) {
        shothits.get(s.id).add(tg);
        // tg.willbehit = true;
        break;
      }
    }
  }

  /**
   * Calculates parameters of all targets.
   */
  private void calcTargets() {
    targets.clear();
    for (List<Target> list : shothits) {
      list.clear();
    }
    for (Asteroid a : state.asteroids) {
      calcTarget(a);
      evalTarget(a);
      targets.add(a);
    }
    final Saucer saucer = state.saucer;
    if (saucer != null) {
      calcTarget(saucer);
      evalTarget(saucer);
      saucer.value -= 30;
      targets.add(saucer);
    }
    Collections.sort(targets);
    for (Shot s : state.shots) {
      final List<Target> list = shothits.get(s.id);
      if (list.size() > 0) {
        shotcomp.setShot(s);
        Collections.sort(list, shotcomp);
        list.get(0).willbehit = true;
      }
    }
  }

  /**
   * Calculates if evasion from hostile shots is needed.
   */
  private void evadeShots() {
    for (Shot s : state.shots) {
      if (ship != null && s.hostile && s.mindist(ship) < ship.radius()
          && s.distance(ship) < ship.radius() + Tables.SHOTSPEED * 2) {
        control.hyperspace(true);
        break;
      }
    }
  }

  /**
   * Evaluates a target as base for target selection.
   * @param tg the target to evaluate
   */
  private void evalTarget(final Target tg) {
    tg.value = 0;
    if (ship == null) {
      return;
    }

    tg.value += Util.abdiff(ship.ab, tg.aimpoint.ab) * 100 / 85;
    tg.value += tg.shipdist * 40 / 512;

    if (tg.willcollide) {
      tg.value -= 20;
    }

    tg.value += tg.speed * 2;
    tg.value -= tg.radius() / 2;
  }

  /**
   * Starts the fire sqeuence.
   * @param ftwo if two shots should be fired
   */
  private void fire(final boolean ftwo) {
    if (firesequence == 0) {
      firesequence++;
      fireseqlen = ftwo ? 4 : 2;
    }
  }

  /**
   * Drives the fire sequence.
   */
  private void firesequence() {
    if (firesequence % 2 == 1) {
      control.fire(true);
    }
    if (firesequence >= fireseqlen) {
      firesequence = 0;
    } else if (firesequence > 0) {
      firesequence++;
    }
  }

  /**
   * Fires at the target if the time is right.
   * @param tg the target to fire at
   */
  private void fireTarget(final Target tg) {
    if (ship == null || tg == null || tg.willbehit || tg.hasbeenshot > 0
        || state.shotsflying >= 4) {
      return;
    }

    boolean ftwo = false;
    if ((tg instanceof Asteroid) && tg.radius() > 10 && state.shotsflying < 3) {
      ftwo = true;
    }

    if (Util.abdiff(ship.ab, tg.aimpoint.ab) <= 1 && tg.speed > 0.5) {
      final Shot ts = testshot(ship.ab);

      if (ts.mindist(tg) <= tg.radius() - 2 && (state.shotsflying < 4)) {
        fire(ftwo);
        target.hasbeenshot = 5;
      }
    }
  }

  /**
   * Returns the testshot with settings for the specified internal angel.
   * @param ab the internal angel
   * @return the testshot
   */
  private Shot testshot(final int ab) {
    testshot.px = ship.px;
    testshot.py = ship.py;
    testshot.vx = 0;
    testshot.vy = 0;
    testshot.setab(ab, ship.vx, ship.vy);
    testshot.px += (int)Math.round(testshot.vx / 1.0);
    testshot.py += (int)Math.round(testshot.vy / 1.0);
    return testshot;
  }

  /**
   * Turn the ship toward the specified target.
   * @param tg the target
   */
  private void turnToTarget(final Target tg) {
    if (ship == null || tg == null) {
      return;
    }
    final int abdiff = Util.abdiff(ship.ab, tg.aimpoint.ab);
    if (abdiff > 0) {
      int ab1 = ship.ab;
      final int ab2 = tg.aimpoint.ab;
      if (ab2 >= 0 && ab2 <= 42 && ab1 >= 236 && ab1 <= 255) {
        ab1 -= 255;
      }
      if (ab2 >= 236 && ab2 <= 255 && ab1 >= 0 && ab1 <= 42) {
        ab1 += 255;
      }
      if (ab2 > ab1 && (abdiff > 1 || !sentleft)) {
        control.left(true);
      } else if (ab2 < ab1 && (abdiff > 1 || !sentright)) {
        control.right(true);
      }
    }
    sentleft = control.isLeft();
    sentright = control.isRight();
  }

  /**
   * Let the bot play.
   */
  synchronized void play() {
    ship = state.ship;
    target = null;
    calcTargets();

    if (targets.size() == 1) {
      target = targets.get(0);
    } else {
      for (Target tg : targets) {
        if (!tg.willbehit && tg.hasbeenshot <= 0) {
          target = tg;
          break;
        }
      }
    }

    turnToTarget(target);
    fireTarget(target);

    evadeShots();
    firesequence();
  }
}
