package de.fhr.asteroids;

import java.awt.Color;

/**
 * A shot from the player or from a hostile saucer.
 * @author Florian Lutz
 * @version 1.1
 */
final class Shot extends GameObject {

  /**
   * The internal angel of the shot.
   */
  int ab = -1;

  /**
   * If the shot is from a hostile saucer.
   */
  boolean hostile;

  /**
   * The id of this shot.
   */
  final int id;

  /**
   * Creates a new shot at the specified position.
   * @param px the x-value of the shot position
   * @param py the y-value of the shot position
   * @param id the id of this shot instance
   */
  private Shot(final int px, final int py, final int id) {
    super(px, py);
    this.id = id;
  }

  /**
   * {@inheritDoc}
   * @see de.fhr.asteroids.Drawable#draw(de.fhr.asteroids.FlipGraphics)
   */
  public void draw(final FlipGraphics g) {
    if (hostile) {
      g.setColor(Color.RED);
    } else {
      g.setColor(Color.ORANGE);
    }
    g.drawCircle(px, py, 4);
    g.drawString(id + "", px + 6, py - 5);
    g.setColor(Color.WHITE);
    g.fillCircle(px, py, 2);
  }

  /**
   * {@inheritDoc}
   * @see de.fhr.asteroids.GameObject#radius()
   */
  @Override
  int radius() {
    return 2;
  }

  /**
   * Sets the internal angle of the shot related to the current speed of the
   * ship.
   * @param nab the new internal angle
   * @param shipvx the x-value of the ships velocity
   * @param shipvy the y-value of the ships velocity
   */
  void setab(final int nab, final double shipvx, final double shipvy) {
    ab = nab;
    if (vx == 0 && vy == 0) {
      vx = Util.sspx(ab, shipvx);
      vy = Util.sspy(ab, shipvy);
      speed = Util.vlen(vx, vy);
    }
  }

  /**
   * Saves the currently used shot ids.
   */
  private static boolean[] usedshotids;

  static {
    usedshotids = new boolean[8];
  }

  /**
   * Factory method to create shots with the correct id.
   * @param px the x-value of the shots position
   * @param py the y-value of the shots position
   * @return the created shot
   */
  static Shot create(final int px, final int py) {
    int id = -1;
    if (px != 0 || py != 0) {
      for (int i = 0; i < usedshotids.length; i++) {
        if (!usedshotids[i]) {
          id = i;
          break;
        }
      }
      usedshotids[id] = true;
    }
    return new Shot(px, py, id);
  }

  /**
   * Removes a shot to free its id.
   * @param shot the shot to remove
   */
  static void remove(final Shot shot) {
    usedshotids[shot.id] = false;
  }
}
