/*
 * Created on Apr 16, 2008
 *
 * (c) 2006-2007 dka - edv, media, webdesign
 *
 */
package com.dkaedv.asteroids.data;

import java.awt.Color;
import java.awt.Graphics2D;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.dkaedv.asteroids.util.VectorCalculations;

public class Position implements IDrawable {
    private ConcurrentLinkedQueue<Vector> vectorQueue = new ConcurrentLinkedQueue<Vector>();
    private Vector curVector;

    private Log log = LogFactory.getLog(Position.class);

    public Position(int curX, int curY) {
        curVector = new Vector(curX, curY);
        vectorQueue.add(curVector);
    }


    /**
     * @return the curVector
     */
    public Vector getCurVector() {
        return curVector;
    }


    /**
     * @return the lastVector
     */
    public Vector getLastVector() {
        return getOldestVector();
    }

    /**
     * Gets a future position.
     *
     * @param ms
     * @return
     */
    public Vector getPredictedVector(long ms) {
        Vector lastVector = getOldestVector();

        // We need a last vector for the calculation
        if (lastVector == null)
            return curVector;

        double targetTimeDiff = System.currentTimeMillis() + ms - curVector.timestamp;

        Vector movingVector = VectorCalculations.getDifference(lastVector, curVector);
        double movingTime = curVector.timestamp - lastVector.timestamp;

        double future_x = ((double)curVector.x) + (
                ((double)movingVector.x) / movingTime * targetTimeDiff
                );

        double future_y = ((double)curVector.y) + (
                ((double)movingVector.y) / movingTime * targetTimeDiff
                );

        Vector predVector = new Vector((int)future_x, (int)future_y);

        //log.debug("Current: " + curVector + " Predicted: " + predVector);
        return predVector;
    }

    /**
     * Returns the speed in coordinates per second.
     *
     * @return
     */
    public double getSpeed() {
        Vector lastVector = getOldestVector();

        if (lastVector == null)
            return 0;

        double movingDistance = VectorCalculations.getDistance(lastVector, curVector);
        double movingTime = curVector.timestamp - lastVector.timestamp;

        return movingDistance / movingTime * 1000;
    }

    public void updatePosition(Vector newVector) {
        curVector = newVector;
        vectorQueue.add(curVector);
    }

    public void draw(Graphics2D g, Graphics2D textG) {
        Vector predVector = getPredictedVector(100);
        g.setColor(Color.YELLOW);
        g.drawLine(curVector.x, curVector.y, predVector.x, predVector.y);
    }

    public double getAngle() {
        Vector v1 = getLastVector();
        Vector v2 = getCurVector();

        Vector diff = VectorCalculations.getDifferenceNormalized(v1, v2);

        return Math.atan2(diff.y, diff.x);
    }

    /*
     * Helpers.
     */
    /**
     * Retrieves the oldest vector from the queue.
     *
     * If the oldest is elder than 200ms it will be removed.
     */
    private Vector getOldestVector() {
        while (true) {
            if (vectorQueue.isEmpty()) {
                return null;
            }

            Vector v = vectorQueue.peek();
            if (System.currentTimeMillis() - v.timestamp > 200) {
                vectorQueue.poll();
            } else {
                return v;
            }
        }
    }

}
