/*
 * Created on Apr 15, 2008
 *
 * (c) 2006-2007 dka - edv, media, webdesign
 *
 */
package com.dkaedv.asteroids.net;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.SocketException;
import java.util.Collections;
import java.util.List;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.dkaedv.asteroids.data.Asteroid;
import com.dkaedv.asteroids.data.GameStatus;
import com.dkaedv.asteroids.data.IPositionable;
import com.dkaedv.asteroids.data.Ship;
import com.dkaedv.asteroids.data.Shot;
import com.dkaedv.asteroids.data.Ufo;
import com.dkaedv.asteroids.util.DistanceComparator;
import com.dkaedv.asteroids.util.VectorCalculations;

public class DatagramReceiver extends Thread {
    private Log log = LogFactory.getLog(DatagramReceiver.class);

    private DatagramSocket socket;
    private GameStatus gameStatus;
    private ConnectionStatus connectionStatus;

    public void run() {
        log.debug("Thread running");

        try {
            while (true) {
                DatagramPacket dgram = new DatagramPacket(new byte[FrameInfo.MAME_DATAGRAM_SIZE], FrameInfo.MAME_DATAGRAM_SIZE);
                socket.receive(dgram);

                //log.debug("Datagram received: " + dgram.getData().toString());

                // Calculate latency
                int receiveSerial = FrameInfo.asInt(dgram.getData()[1025]); // This is the Ping byte
                connectionStatus.calculateLatency(receiveSerial, System.currentTimeMillis());

                // TODO Dropped frames calculation

                // Interpret datagram
                DisplayDatagram displayDatagram = interpretDatagram(dgram.getData());
                displayDatagram.setReceiveTimestamp(System.currentTimeMillis());
                updateGameStatus(displayDatagram);

                // Call listeners
                gameStatus.wakeUpWaiters();
            }

        } catch (SocketException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    /*
     * Helper functions.
     */

    DisplayDatagram interpretDatagram(byte[] data) {
        DisplayDatagram dgram = new DisplayDatagram();

        int[] vectorRamWords = new int[FrameInfo.VECTOR_RAM_SIZE / 2];
        for (int r = 0, w = 0; w < 512; r += 2, w++) {
            vectorRamWords[w] =
                (FrameInfo.asInt(data[r])) | (FrameInfo.asInt(data[r + 1]) << 8);
        }


        // Interpret datagram
        int gsf = 0, pos_x = 0, pos_y = 0, dy = 0, dx = 0, sf = 0, vz = 0, v1x = 0, v1y = 0, shipdetect = 0;

        for (int i = 0; i < vectorRamWords.length; i++) {
            int opcode = vectorRamWords[i] >> 12;
            switch (opcode) {
            case FrameInfo.OPCODE_LABS: // Strahl positionieren
                pos_y = vectorRamWords[i] & 0x3ff; // rechten 10 Bits
                pos_x = vectorRamWords[i+1] & 0x3ff; // rechten 10 Bits
                gsf = vectorRamWords[i+1] >> 12; // GSF - Globaler Skalierungsfaktor
                //log.debug("OPCODE_LABS X:" + pos_x + " Y:" + pos_y + " GSF:" + gsf);
                break;
            case FrameInfo.OPCODE_HALT: // End of datagram
                return dgram;
            case FrameInfo.OPCODE_JSRL: // Jump to subroutine
                switch (vectorRamWords[i] & 0xfff) // rechten 12 Bits
                {
                case FrameInfo.JSRL_ASTEROID_TYPE_1:
                    dgram.getAsteroids().add(new Asteroid(pos_x, pos_y, 1, gsf));
                    break;
                case FrameInfo.JSRL_ASTEROID_TYPE_2:
                    dgram.getAsteroids().add(new Asteroid(pos_x, pos_y, 2, gsf));
                    break;
                case FrameInfo.JSRL_ASTEROID_TYPE_3:
                    dgram.getAsteroids().add(new Asteroid(pos_x, pos_y, 3, gsf));
                    break;
                case FrameInfo.JSRL_ASTEROID_TYPE_4:
                    dgram.getAsteroids().add(new Asteroid(pos_x, pos_y, 4, gsf));
                    break;
                case FrameInfo.JSRL_SHIP:
                    dgram.setLives(dgram.getLives() + 1);
                    break;
                case FrameInfo.JSRL_UFO:
                    dgram.setUfo(new Ufo(pos_x, pos_y, gsf));
                    break;
                }
                break;
            case FrameInfo.OPCODE_RTSL:
                // Return from subroutine can be safely ignored for now
                break;
            case FrameInfo.OPCODE_JMPL:
                // Jumps can be safely ignored for now
                break;
            case FrameInfo.OPCODE_SVEC:
                /*
                dy = vectorRamWords[pc] & 0x300;
                if ((vectorRamWords[pc] & 0x400) != 0)
                    dy = -dy;
                dx = (vectorRamWords[pc] & 3) << 8;
                if ((vectorRamWords[pc] & 4) != 0)
                    dx = -dx;
                sf = (((vectorRamWords[pc] & 8) >> 2) | ((vectorRamWords[pc] & 0x800) >> 11)) + 2;
                vz = (vectorRamWords[pc] & 0xf0) >> 4;
                */
                break;
            default: // Draw a long vector
                dy = vectorRamWords[i] & 0x3ff;
                if ((vectorRamWords[i] & 0x400) != 0)
                    dy = -dy;
                dx = vectorRamWords[i+1] & 0x3ff;
                if ((vectorRamWords[i+1] & 0x400) != 0)
                    dx = -dx;
                sf = opcode;
                vz = vectorRamWords[i+1] >> 12; // Helligkeit

                // Shot
                if (dx == 0 && dy == 0 && vz == 15)
                    dgram.getShots().add(new Shot(pos_x, pos_y));

                // Ship
                if (opcode == 6 && vz == 12 && dx != 0 && dy != 0)
                {
                    switch (shipdetect)
                    {
                    case 0:
                        v1x = dx;
                        v1y = dy;
                        ++shipdetect;
                        break;
                    case 1:
                        dgram.setShip(new Ship(pos_x, pos_y, v1x - dx, v1y - dy));
                        ++shipdetect;
                        break;
                    }
                }
                else if (shipdetect == 1)
                    shipdetect = 0;

                break;
            }

            // All commands < 0xA are 2 word, rest 1 word
            if (opcode <= 0xa) {
                i++;
            }
        }

        return dgram;
    }

    private void updateGameStatus(DisplayDatagram dgram) {
        // Copy the list of current asteroids
        List<Asteroid> currentAsteroids = new Vector<Asteroid>();
        currentAsteroids.addAll(gameStatus.getAsteroids());

        for (Asteroid a : dgram.getAsteroids()) {
            if (currentAsteroids.size() > 0) {

                Collections.sort(currentAsteroids, new DistanceComparator(a.getPosition().getCurVector()));

                Asteroid updateAsteroid = currentAsteroids.get(0);
                if (updateAsteroid.getScaleFactor() == a.getScaleFactor()
                        && updateAsteroid.getType() == a.getType()) {

                    // Ok, this asteroid fits
                    updateAsteroid.getPosition().updatePosition(a.getPosition().getCurVector());

                    // Remove asteroid from update list
                    currentAsteroids.remove(updateAsteroid);
                } else {
                    // Closest asteroid doesn't fit
                    gameStatus.getAsteroids().add(a);
                }

            } else {
                // No asteroids left
                gameStatus.getAsteroids().add(a);
            }
        }

        // Remove any current asteroids that have not been updated
        gameStatus.getAsteroids().removeAll(currentAsteroids);

        // Update ship
        if (gameStatus.getShip() != null
                && dgram.getShip() != null) {
            gameStatus.getShip().getPosition().updatePosition(dgram.getShip().getPosition().getCurVector());
            gameStatus.getShip().setDx(dgram.getShip().getDx());
            gameStatus.getShip().setDy(dgram.getShip().getDy());
        } else {
            gameStatus.setShip(dgram.getShip());
        }

        if (gameStatus.getUfo() != null
                && dgram.getUfo() != null) {
            gameStatus.getUfo().getPosition().updatePosition(dgram.getUfo().getPosition().getCurVector());
        } else {
            gameStatus.setUfo(dgram.getUfo());
        }

        // Copy the list of current shots
        List<Shot> currentShots = new Vector<Shot>();
        currentShots.addAll(gameStatus.getShots());

        double avgShotSpeed = gameStatus.getAverageShotSpeed();

        for (Shot s : dgram.getShots()) {
            Collections.sort(currentShots, new DistanceComparator(s.getPosition().getCurVector()));

            boolean updatedShot = false;


            if (currentShots.size() > 0) {
                List<Shot> currentShotsDup = new Vector<Shot>(currentShots);
                for (Shot updateShot : currentShotsDup) {
                    if ( !updatedShot ) {
                        // No shot has been updated so far

                        double shotDistance = VectorCalculations.getDistance(updateShot, s);
                        double timeDiff = s.getPosition().getCurVector().timestamp - updateShot.getPosition().getCurVector().timestamp;

                        if (updateShot.getType() == Shot.TYPE_SHIP
                            && shotDistance < (avgShotSpeed / 1000.0 * timeDiff * 1.5)
                            && shotDistance > (avgShotSpeed / 1000.0 * timeDiff * 0.5)
                            ) {

                            updateShot.getPosition().updatePosition(s.getPosition().getCurVector());

                            // Remove shot from update list
                            currentShots.remove(updateShot);
                            updatedShot = true;
                        } else if (updateShot.getType() == Shot.TYPE_UFO
                                && shotDistance < 20
                            ) {

                            updateShot.getPosition().updatePosition(s.getPosition().getCurVector());

                            // Remove shot from update list
                            currentShots.remove(updateShot);
                            updatedShot = true;
                        }
                    }
                }
            }

            if (!updatedShot) {
                // No shots left
                gameStatus.getShots().add(s);

                if (gameStatus.getUfo() != null
                        && gameStatus.getShip() != null) {
                    double distanceToShip = VectorCalculations.getDistance(gameStatus.getShip(), s);
                    double distanceToUfo = VectorCalculations.getDistance(gameStatus.getUfo(), s);

                    //log.debug("distanceShip: " + distanceToShip + " distanceUfo: " + distanceToUfo);

                    if (distanceToUfo < distanceToShip
                            && distanceToUfo < 30) {
                        s.setType(Shot.TYPE_UFO);
                    } else {
                        s.setType(Shot.TYPE_SHIP);
                    }
                } else if (gameStatus.getShip() != null) {
                    // No UFO present, so this is a ship shot
                    s.setType(Shot.TYPE_SHIP);
                } else if (gameStatus.getUfo() != null) {
                    // No ship present
                    s.setType(Shot.TYPE_UFO);
                }
            }
        }

        // Remove any current shots that have not been updated
        gameStatus.getShots().removeAll(currentShots);

        // Update numbers
        gameStatus.setLives(dgram.getLives());
        gameStatus.setReceiveTimestamp(dgram.getReceiveTimestamp());
    }

    /*
     * Injected spring beans.
     */

    /**
     * @param socket the socket to set
     */
    public void setSocket(DatagramSocket socket) {
        this.socket = socket;
    }


    /**
     * @param gameStatus the gameStatus to set
     */
    public void setGameStatus(GameStatus gameStatus) {
        this.gameStatus = gameStatus;
    }


    /**
     * @param connectionStatus the connectionStatus to set
     */
    public void setConnectionStatus(ConnectionStatus connectionStatus) {
        this.connectionStatus = connectionStatus;
    }
}
