/*****************************************************************************
*  ADBotPlayerV1                                                             *
*                                                                            *
*  Copyright (C) 2008 Andreas Dittrich                                       *
*                                                                            *
*  Author:  Andreas Dittrich                                                 *
*           Am Schimmelberg 29                                               *
*           67729 Sippersfeld (Germany)                                      *
*****************************************************************************/

#ifndef VECTOR_H
#define VECTOR_H

#include <mmintrin.h>


#define INLINE __forceinline


#define NUM_BITS_FRACTION 3
#define UNIT (1<<NUM_BITS_FRACTION)

#define WINDOW_WIDTH (1024<<NUM_BITS_FRACTION)
#define WINDOW_HEIGHT (768<<NUM_BITS_FRACTION)
#define Y_POS_OFFSET  (128<<NUM_BITS_FRACTION)

#ifndef VECTOR_T
	#define VECTOR_T
	typedef struct {
		int x;
		int y;
	} vector_t;
#endif

/*
static INLINE int ABS(int x)
{
	x -= ((x>>31)&(x<<1));
}
*/

static INLINE int MIN(int a, int b)
{
	if (a<b)
		return a;

	return b;
}


static INLINE int MAX(int a, int b)
{
	if (a>b)
		return a;

	return b;
}


static INLINE int normalize_dx( int dx )
{
	while (dx < -(WINDOW_WIDTH/2)) 
		dx += WINDOW_WIDTH;

	while (dx >= (WINDOW_WIDTH/2)) 
		dx -= WINDOW_WIDTH;

	return dx;
}


static INLINE int normalize_dy( int dy )
{
	while (dy < -(WINDOW_HEIGHT/2)) 
		dy += WINDOW_HEIGHT;

	while (dy >= (WINDOW_HEIGHT/2)) 
		dy -= WINDOW_HEIGHT;

	return dy;
}


static INLINE void normalize_d( vector_t *pd )
{
	pd->x = normalize_dx( pd->x );
	pd->y = normalize_dy( pd->y );
}




static INLINE int normalize_rx( int rx )
{
	//while (rx < 0) 
	//	rx += WINDOW_WIDTH;

	//while (rx >= WINDOW_WIDTH) 
	//	rx -= WINDOW_WIDTH;

	return ( rx & 0x1FFF );
}


static INLINE int normalize_ry( int ry )
{
	if ( (unsigned int)(ry-Y_POS_OFFSET) < WINDOW_HEIGHT )
		return ry;

	ry -= Y_POS_OFFSET;

	if (ry<0)
	{
		do {
			ry += WINDOW_HEIGHT;
		} while (ry<0);

		return ry + Y_POS_OFFSET;
	}

	ry -= WINDOW_HEIGHT;

	if (ry>=0)
	{
		do {
			ry -= WINDOW_HEIGHT;
		} while (ry>=0);

		return ( ry + Y_POS_OFFSET + WINDOW_HEIGHT );
	}

	return 0;
}


static INLINE void normalize_r( vector_t *pr )
{
	pr->x = normalize_rx( pr->x );
	pr->y = normalize_ry( pr->y );
}


static INLINE void vsub( vector_t *pa, vector_t b )
{
	pa->x -= b.x;
	pa->y -= b.y;
}


static INLINE void vsub_mmx( vector_t *pa, vector_t b )
{
	__m64 mm0 = *(__m64*)pa;
	*(__m64*)pa = _mm_sub_pi32( mm0, *(__m64*)(&b) );
}


static INLINE void vadd( vector_t *pa, vector_t b )
{
	pa->x += b.x;
	pa->y += b.y;
}


static INLINE void vadd_mmx( vector_t *pa, vector_t b )
{
	__m64 mm0 = *(__m64*)pa;
	*(__m64*)pa = _m_paddd( mm0, *(__m64*)(&b) );
}


static INLINE int vscalarprod( vector_t a, vector_t b )
{
	return a.x*b.x + a.y*b.y;
}


static INLINE int vabsq( vector_t a )
{
	return vscalarprod( a, a );
}


static INLINE void vmul( vector_t *pv, int k )
{
	pv->x *= k;
	pv->y *= k;
}


static INLINE void vsar( vector_t *pb, uint8_t n )
{
	pb->x >>= n;
	pb->y >>= n;
}


static INLINE void vshl( vector_t *pb, uint8_t n )
{
	pb->x <<= n;
	pb->y <<= n;
}


static INLINE void vmuladd( vector_t *pb, vector_t a, int k )
{
	pb->x += k * a.x;
	pb->y += k * a.y;
}


static INLINE void vmulsub( vector_t *pb, vector_t a, int k )
{
	pb->x -= k * a.x;
	pb->y -= k * a.y;
}




static INLINE int is_hit_octagon( 
	vector_t a, 
	int width )
{
	const int dx = ( a.x >> 1 )^( a.x >> 31 );
	const int dy = ( a.y >> 1 )^( a.y >> 31 );

	//the last bits may be not accurate
//	dx += 3;
//	dy += 3;

	if ( ( (width-dx) | (width-dy) ) < 0 )
		return 0;

	if ( (width+(width>>1)-dx-dy) <= 0 )
		return 0;

	return 1;
}


static const int asteroid_width_table[8] =
{
	0,
	0x2A,
	0x48,
	0,
	0x84,
	0,
	0,
	0x25
};


int INLINE is_hit_shot( vector_t a, int a_state )
{
	const int width = asteroid_width_table[a_state&0x07];

	return is_hit_octagon( a, width );
}


static const int ship_width_table[8] =
{
	0,
	0x2A + 0x1C,
	0x48 + 0x1C,
	0,
	0x84 + 0x1C,
	0,
	0,
	0x25 + 0x1C
};


int INLINE is_collision_asteroid_ship( vector_t a, int a_state )
{

	const int width = ship_width_table[a_state&0x07];

	return is_hit_octagon( a, width );
}



#endif