## ----setup, include=FALSE--------------------------------------------------
knitr::opts_chunk$set(
  collapse=TRUE,
  comment="#>",
  warning=FALSE,
  message=FALSE,
  error=FALSE
)

## ----library---------------------------------------------------------------
library(BiocStyle)
library(HPAanalyze)
library(tibble)
library(dplyr)
library(ggplot2)
library(hpar)

## ----downloadedData, eval=FALSE--------------------------------------------
#  downloadedData <- hpaDownload(downloadList='all')
#  summary(downloadedData)
#  
#  #>                          Length Class  Mode
#  #> normal_tissue             6     tbl_df list
#  #> pathology                11     tbl_df list
#  #> subcellular_location     11     tbl_df list
#  #> rna_tissue                5     tbl_df list
#  #> rna_cell_line             5     tbl_df list
#  #> transcript_rna_tissue     4     tbl_df list
#  #> transcript_rna_cell_line  4     tbl_df list

## ----normal_tissue---------------------------------------------------------
downloadedData <- hpaDownload(downloadList='histology', version='example')
tibble::glimpse(downloadedData$normal_tissue, give.attr=FALSE)

## ----pathology-------------------------------------------------------------
tibble::glimpse(downloadedData$pathology, give.attr=FALSE)

## ----subcellular_location--------------------------------------------------
tibble::glimpse(downloadedData$subcellular_location, give.attr=FALSE)

## ----rna1, warning=FALSE, message=FALSE------------------------------------
downloadedData <- hpaDownload(downloadList='rna', version='hpar')

tibble::glimpse(downloadedData$rna_tissue, give.attr=FALSE)

## ----rna2, warning=FALSE, message=FALSE------------------------------------
tibble::glimpse(downloadedData$rna_cell_line, give.attr=FALSE)

## ----isoform1, eval=FALSE--------------------------------------------------
#  downloadedData <- hpaDownload(downloadList='isoform', version='v18')
#  
#  tibble::glimpse(downloadedData$transcript_rna_tissue, give.attr=FALSE)
#  
#  #> Observations: 27,535,996
#  #> Variables: 4
#  #> $ ensembl    <chr> "ENSG00000000003", "ENSG00000000003", "ENSG0000000...
#  #> $ transcript <chr> "ENST00000373020", "ENST00000494424", "ENST0000049...
#  #> $ tissue     <chr> "adipose tissue.V1", "adipose tissue.V1", "adipose...
#  #> $ value      <dbl> 27.3577003, 0.0000000, 1.9341500, 1.6059300, 0.000...

## ----isoform2, eval=FALSE--------------------------------------------------
#  tibble::glimpse(downloadedData$transcript_rna_cell_line, give.attr=FALSE)
#  
#  #> Observations: 20,972,183
#  #> Variables: 4
#  #> $ ensembl    <chr> "ENSG00000000003", "ENSG00000000003", "ENSG0000000...
#  #> $ transcript <chr> "ENST00000373020", "ENST00000494424", "ENST0000049...
#  #> $ cell_line  <chr> "A-431.C35", "A-431.C35", "A-431.C35", "A-431.C35"...
#  #> $ value      <dbl> 29.406799, 0.000000, 0.992916, 0.398387, 0.239204,...

## ----hpar1, eval=FALSE-----------------------------------------------------
#  hpaDownload('all', 'hpar')

## ----hpar2-----------------------------------------------------------------
hpar::getHpaVersion()

hpar::getHpaDate()

hpar::getHpaEnsembl()

## ----list_param, eval=FALSE------------------------------------------------
#  downloadedData <- hpaDownload(downloadList='all')
#  str(hpaListParam(downloadedData))
#  
#  #> List of 6
#  #>  $ normal_tissue       : chr [1:58] "adrenal gland" "appendix" "bone marrow" "breast" ...
#  #>  $ normal_cell         : chr [1:82] "glandular cells" "lymphoid tissue" "hematopoietic cells" "adipocytes" ...
#  #>  $ cancer              : chr [1:20] "breast cancer" "carcinoid" "cervical cancer" "colorectal cancer" ...
#  #>  $ subcellular_location: chr [1:32] "Cytosol" "Mitochondria" "Aggresome" "Plasma membrane" ...
#  #>  $ normal_tissue_rna   : chr [1:37] "adipose tissue" "adrenal gland" "appendix" "bone marrow" ...
#  #>  $ cell_line_rna       : chr [1:64] "A-431" "A549" "AF22" "AN3-CA" ...

## ----subset1, message=FALSE, warning=FALSE---------------------------------
downloadedData <- hpaDownload(downloadList='histology', version='example')
sapply(downloadedData, nrow)

## ----subset2, message=FALSE, warning=FALSE---------------------------------
geneList <- c('TP53', 'EGFR', 'CD44', 'PTEN', 'IDH1', 'IDH2', 'CYCS')
tissueList <- c('breast', 'cerebellum', 'skin 1')
cancerList <- c('breast cancer', 'glioma', 'melanoma')
cellLineList <- c('A-431', 'A549', 'AF22', 'AN3-CA')

subsetData <- hpaSubset(data=downloadedData,
                         targetGene=geneList,
                         targetTissue=tissueList,
                         targetCancer=cancerList,
                         targetCellLine=cellLineList)
sapply(subsetData, nrow)

## --------------------------------------------------------------------------
id <- c("ENSG00000000003", "ENSG00000000005")
hpar::getHpa(id, hpadata="hpaNormalTissue") %>%
    tibble::glimpse()

## ----eval=FALSE------------------------------------------------------------
#  hpaExport(subsetData, fileName='subset.xlsx', fileType='xlsx')

## ----visData, echo=FALSE, warning=FALSE, message=FALSE---------------------
downloadedData <- hpaDownload('histology', 'example')

## ----visTissue, fig.wide---------------------------------------------------
geneList <- c('TP53', 'EGFR', 'CD44', 'PTEN', 'IDH1', 'IDH2', 'CYCS')
tissueList <- c('breast', 'cerebellum', 'skin 1')

hpaVisTissue(downloadedData,
             targetGene=geneList,
             targetTissue=tissueList)

## ----visPatho, fig.wide----------------------------------------------------
geneList <- c('TP53', 'EGFR', 'CD44', 'PTEN', 'IDH1', 'IDH2', 'CYCS')
cancerList <- c('breast cancer', 'glioma', 'lymphoma', 'prostate cancer')
colorGray <- c('slategray1', 'slategray2', 'slategray3', 'slategray4')

hpaVisPatho(downloadedData,
            targetGene=geneList,
            targetCancer=cancerList,
            color=colorGray)

## ----visSubcell------------------------------------------------------------
geneList <- c('TP53', 'EGFR', 'CD44', 'PTEN', 'IDH1', 'IDH2', 'CYCS')

hpaVisSubcell(downloadedData,
              targetGene=geneList,
              customTheme=TRUE) +
    ggplot2::theme_minimal() +
    ggplot2::ylab('Subcellular locations') +
    ggplot2::xlab('Protein') +
    ggplot2::theme(axis.text.x=element_text(angle=45, hjust=1))  +
    ggplot2::theme(legend.position="none") +
    ggplot2::coord_equal()

## ----XmlGet----------------------------------------------------------------
CCNB1xml <- hpaXmlGet('ENSG00000134057')

## ----hpar_site, eval=FALSE-------------------------------------------------
#  hpar::getHpa('ENSG00000134057', type="details")

## ----XmlProtClass----------------------------------------------------------
hpaXmlProtClass(CCNB1xml)

## ----XmlTissueExprSum------------------------------------------------------
hpaXmlTissueExprSum(CCNB1xml)

## ----XmlAntibody-----------------------------------------------------------
hpaXmlAntibody(CCNB1xml)

## ----XmlTissueExpr1--------------------------------------------------------
tissueExpression <- hpaXmlTissueExpr(CCNB1xml)
summary(tissueExpression)

## ----XmlTissueExpr2--------------------------------------------------------
tissueExpression[[1]]

## ----echo=FALSE------------------------------------------------------------
sessionInfo()

