--==========================================================================--
-- Design units : 10/100 Ethernet MAC - RxMAC - RxLengthCounter
--
-- File name    : rxlengthcounter.vhd
--
-- Purpose      : The RxLengthCounter counts the number of nibbles during
--                frame reception. Counting is done when RxValid is one. The
--                rising edge of this signal resets the counter to zero. So
--                at each clock cycle the value of the counter shows the
--                offset of the current nibble in the RxDataRegister within
--                the frame. After reception the value is equal to the total
--                length of the frame.
--
-- Note         : None
--
-- Limitations  : None
--
-- Errors       : None known
--
-- Library      : EthMAC_Lib
--
-- Dependencies : None
--
-- Author       : Maik Boden (boden@ite.inf.tu-dresden.de)
--                Dresden University of Technology
--                Department of Information Science
--                Institute of Computer Engineering
--
-- Simulator    : VHDL Simili 1.4 Beta, Windows NT 4.0
------------------------------------------------------------------------------
-- Revision list
-- Version  Author  Date        Changes
-- 0.1      MB      2000-09-04  Initial revision
--          MB      2000-09-19  RxLate added
--          MB      2000-11-28  IEEE.Std_Logic_Arith used to describe on RTL
--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;

entity RxLengthCounter is
  port(
    -- Receive clock and asynchronous reset
    RxReset_N:   in  Std_ULogic;
    RxClk:       in  Std_ULogic;
    -- Valid frame data indicator
    RxValid:     in  Std_ULogic;
    -- Offset of the current nibble within the frame
    RxLate:      out Std_ULogic;
    RxLength:    out Std_ULogic_Vector(11 downto 0)
  );
end RxLengthCounter;

--==========================================================================--

library IEEE;
use IEEE.Std_Logic_1164.all;
use IEEE.Std_Logic_Arith.all;

architecture RTL of RxLengthCounter is

  ----------------------------------------------------------------------------
  -- Internal 12-bit-wide register to count number of received nibbles.
  ----------------------------------------------------------------------------
  signal RxLenReg: Unsigned(11 downto 0) := (others => '0');

begin  -- Architecture RTL of RxLengthCounter

  ----------------------------------------------------------------------------
  -- If RxValid is asserted the length register is increased by one on each
  -- clock cycle to count the offset of the current nibble within the frame.
  -- Deassertion of RxValid resets register within the next clock cycle.
  ----------------------------------------------------------------------------
  CountLength: process (RxReset_N, RxClk)
  begin
    if RxReset_N = '0' then
      RxLenReg <= (others => '0');
    elsif Rising_Edge(RxClk) then
      if RxValid = '1' then
        RxLenReg <= RxLenReg + 1;
      else
        RxLenReg <= (others => '0');
      end if;
    end if;
  end process CountLength;

  ----------------------------------------------------------------------------
  -- Propagate contents of counter to output RxLength.
  ----------------------------------------------------------------------------
  RxLength <= To_StdULogicVector(Conv_Std_Logic_Vector(RxLenReg, 12));

  ----------------------------------------------------------------------------
  -- RxLate is set when first slot-time is over and then any collision
  -- indicates a fatal network error.
  ----------------------------------------------------------------------------
  RxLate <= RxLenReg(11) or RxLenReg(10) or RxLenReg(9) or RxLenReg(8)
            or RxLenReg(7) or (RxLenReg(6) and RxLenReg(5) and RxLenReg(4)
            and RxLenReg(3) and RxLenReg(2) and RxLenReg(1) and RxLenReg(0));

end RTL;  -- End of RxLengthCounter (RTL)
